package com.xebialabs.xlrelease.authentication

import com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty
import com.xebialabs.deployit.plumbing.authorization.LoginAuthorizationManager
import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.deployit.security.permission.PlatformPermissions
import com.xebialabs.xlrelease.security.authentication.policy.{UserGroupMembershipCreationPolicy, UserProfileCreationPolicy}
import org.springframework.context.ApplicationListener
import org.springframework.security.authentication.DisabledException
import org.springframework.security.authentication.event.AuthenticationSuccessEvent
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Component
class AuthenticationSuccessEventListener(userProfileCreationPolicies: JList[UserProfileCreationPolicy],
                                         userGroupMembershipCreationPolicies: JList[UserGroupMembershipCreationPolicy],
                                         permissionEnforcer: PermissionEnforcer
                                        )
  extends ApplicationListener[AuthenticationSuccessEvent] {

  override def onApplicationEvent(event: AuthenticationSuccessEvent): Unit = {
    val authentication: Authentication = LoginAuthorizationManager.addAuthenticatedUserAuthority(event.getAuthentication)
    if (null != authentication && isNotEmpty(authentication.getName)) {
      val hasLoginPermission = permissionEnforcer.hasPermission(authentication, PlatformPermissions.LOGIN)
      if (!hasLoginPermission) {
        throw new DisabledException(s"The user ${authentication.getName} or its group(s) do not have login permission")
      }
      val creationPolicy = findApplicableUserCreationPolicy(authentication)
      creationPolicy.createProfile(authentication)

      val ugCreationPolicy = findApplicableUserGroupCreationPolicy(authentication)
      ugCreationPolicy.updateUserGroupMembership(authentication)
    }
  }

  private def findApplicableUserCreationPolicy(authentication: Authentication) =
    userProfileCreationPolicies.asScala
      .sortBy(_.order())
      .find(_.policyApplies(authentication))
      .getOrElse(throw new IllegalStateException(s"No UserProfile creation policy applicable for authentication [$authentication]"))

  private def findApplicableUserGroupCreationPolicy(authentication: Authentication) =
    userGroupMembershipCreationPolicies.asScala
      .sortBy(_.order())
      .find(_.policyApplies(authentication))
      .getOrElse(throw new IllegalStateException(s"No UserGroupMembership creation policy applicable for authentication [$authentication]"))
}
