package com.xebialabs.xlrelease.script.builder

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.CustomScriptTask
import com.xebialabs.xlrelease.script.{EncryptionHelper, ScriptTimeoutContext, XlrScript, XlrScriptContext}

import java.util.concurrent.ScheduledExecutorService
import scala.jdk.CollectionConverters._
import scala.jdk.DurationConverters._


class CustomScriptTaskScriptContextBuilder(xlrConfig: XlrConfig, task: CustomScriptTask, timeoutExecutor: ScheduledExecutorService)
  extends ScriptContextBuilder {
  withCustomScriptApi().withLogger().withExecutionId(task.getExecutionId)
  val timeoutEnabled = xlrConfig.timeouts.customScriptTaskTimeoutEnabled
  val timeout = xlrConfig.timeouts.customScriptTaskTimeout.toJava
  val settingsKey = xlrConfig.timeouts.CUSTOM_SCRIPT_TASK_TIMEOUT_KEY
  withTimeoutContext(new ScriptTimeoutContext(timeoutEnabled, timeoutExecutor, timeout, settingsKey))

  task.getEngine match {
    case "groovy" => logger.warn("Groovy engine is not supported for CustomScriptTasks")
    case "jython" => withPythonSugar().withPythonGlobals().withPythonReleaseApi().withPythonUtilities()
  }

  override protected def doBuild(context: XlrScriptContext): Unit = {
    EncryptionHelper.decrypt(task.getRelease)
    EncryptionHelper.decrypt(task.getPythonScript)
    val pythonScript = task.getPythonScript
    val inputProperties = pythonScript.getInputProperties.asScala
    val transitionalProperties = pythonScript.getTransitionalProperties.asScala
    val outputProperties = if (task.hasNextScriptToExecute) {
      pythonScript.getOutputProperties.asScala
    } else {
      Seq()
    }
    val ctxProperties = inputProperties ++ transitionalProperties ++ outputProperties
    context.addDomainObjects(task)
    context.setScriptPathForLog(task.getScriptPath)
    context.addProperties(pythonScript, ctxProperties.asJavaCollection)
    context.addPropertyBindings(pythonScript.getTransitionalAndOutputProperties)

    val scriptPath = if (task.isAbortScriptInProgress) {
      task.getAbortScriptPath
    } else {
      task.getScriptPath
    }
    val script = XlrScript.byResource(s"${task.getType}[${task.getId}]", scriptPath, wrap = true, checkPermissions = false)
    context.addScript(script)
  }
}
