package com.xebialabs.xlrelease.actors

import org.apache.pekko.actor.{Actor, ActorPath, ActorRef, ActorSystem}
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout
import org.slf4j.LoggerFactory

import scala.concurrent.duration.Duration
import scala.concurrent.{Await, Future, Promise, TimeoutException}

trait ManagedActor[T] {

  private def logger = LoggerFactory.getLogger(classOf[ManagedActor[_]])

  private var actorPromise = Promise[ActorRef]()

  def resolveTimeout(): Duration

  def responseTimeout(): Timeout

  private def actorRef(): ActorRef = {
    Await.result(actorPromise.future, resolveTimeout())
  }

  private[actors] def reset(): Unit = {
    logger.trace(s"Resetting managed actor for '{}'", name())
    actorPromise = Promise[ActorRef]()
  }

  //noinspection ScalaStyle
  def !(message: Any)(implicit sender: ActorRef = Actor.noSender): Unit = {
    warnAboutUninitializedManagedActor()
    actorRef() ! message
  }

  //noinspection ScalaStyle
  def ?(message: Any)(implicit timeout: Timeout, sender: ActorRef = Actor.noSender): Future[Any] = {
    warnAboutUninitializedManagedActor()
    actorRef() ? message
  }

  def path: ActorPath = {
    actorRef().path
  }

  private def warnAboutUninitializedManagedActor(): Unit = {
    if (!actorPromise.isCompleted) {
      logger.warn(s"Sending message to uninitialized managed actor for '{}'", name())
    }
  }

  protected def actorCreator: ActorCreator

  def name(): String

  def initialize(): Unit = {
    if (!actorPromise.isCompleted) {
      logger.info(s"Creating managed actor for '{}'", name())
      actorPromise.success(actorCreator.create(getActorSystem))
    } else {
      logger.trace(s"Managed actor for '{}' is already created", name())
    }
  }

  def isInitialized: Boolean = {
    actorPromise.isCompleted
  }

  def getActorSystem: ActorSystem

  def askAndAwait[R](msg: AnyRef): R = {
    logger.trace(s"${name()}: askAndAwait ${msg.toString}")
    val result = try {
      Await.result(this.?(msg)(responseTimeout()),responseTimeout().duration).asInstanceOf[R]
    } catch {
      case e: TimeoutException =>
        logger.error(s"${name()}: got TimeoutException in askAndAwait for ${msg.toString}", e)
        throw new RuntimeException(e)
      case e: InterruptedException =>
        logger.error(s"${name()}: Got InterruptedException in askAndAwait for ${msg.toString}", e)
        throw new RuntimeException(e)
      case e: Throwable =>
        logger.error(s"${name()}: Got exception in askAndAwait for ${msg.toString}", e)
        throw e
    }
    logger.trace(s"${name()}: askAndAwait resolved ${msg.toString} to $result")
    result
  }
}
