package com.xebialabs.xlrelease.actors

import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.full.downing.AutoDowning
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.DatabaseInfo
import com.xebialabs.xlrelease.db.sql.DatabaseInfo._
import com.xebialabs.xlrelease.support.pekko.GracefulShutdownReason
import com.xebialabs.xlrelease.support.pekko.spring._
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{Actor, ActorRef, ActorSystem, CoordinatedShutdown, PoisonPill, Props}
import org.apache.pekko.cluster.Cluster
import org.apache.pekko.cluster.sharding.{ClusterSharding, ClusterShardingSettings, ShardRegion}
import org.apache.pekko.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings, ClusterSingletonProxy, ClusterSingletonProxySettings}
import org.apache.pekko.management.scaladsl.PekkoManagement
import org.apache.pekko.util.Timeout
import org.springframework.beans.factory.annotation.Qualifier
import slick.jdbc._

import scala.collection.mutable
import scala.concurrent.Await
import scala.concurrent.duration.Duration
import scala.jdk.CollectionConverters._

class ActorSystemHolder(xlrConfig: XlrConfig, @Qualifier("xlrDatabaseInformation") dbInfo: DatabaseInfo)
  extends Logging
    with ScalaSpringAwareBean
    with ActorFactory {

  private var running: Boolean = false

  private var _actorSystem: ActorSystem = _

  private def getInstance(): ActorSystem = {
    if (isActorSystemInitialized()) {
      _actorSystem
    } else {
      throw new IllegalStateException("Actor system is not initialized")
    }
  }

  def createActorSystem(): ActorSystem = {
    if (running) {
      throw new IllegalStateException("Actor system is already initialized")
    }
    _actorSystem = doCreateActorSystem()
    running = true
    _actorSystem
  }

  protected def doCreateActorSystem(): ActorSystem = {
    info("Starting up actor system.")
    if (xlrConfig.isActiveOnStartup) {
      NodeState.setActive(true)
    }

    val pekkoConfig = getSlickConfig.withFallback(xlrConfig.pekko.config)
    val system = ActorSystem(xlrConfig.pekko.systemName, pekkoConfig)
    system
  }

  //If spring finds a shutdown method on bean, it will call it - we do not want it here
  def stop(): Unit = {
    if (running) {
      info("Initiating actor system shutdown.")
      if (!AutoDowning.isDowning) {
        // Actor system should be terminated by Coordinated Shutdown Pekko extension
        CoordinatedShutdown.get(_actorSystem).run(GracefulShutdownReason)
        info("Waiting for actor system shutdown (indefinitely).")

        val result = _actorSystem.whenTerminated
        Await.result(result, atMost = Duration.Inf)
      }
      running = false
      _actorSystem = null
      info("Actor system shutdown finished.")
    }
  }

  def isActorSystemInitialized(): Boolean = running && _actorSystem != null

  @deprecated("This method should be used with caution, and it is there to support legacy use-case where actor system restart was not implemented.", "25.1.0")
  def unmanagedActorSystem: ActorSystem = getInstance()

  def actorSystemName(): String = getInstance().name

  def cluster(): Cluster = Cluster(getInstance())

  def pekkoManagement(): PekkoManagement = PekkoManagement(getInstance())

  //get slick config for pekko-persistence-jdbc
  private def getSlickConfig: Config = {
    val slickProfileConfigKey: String = "pekko-persistence-jdbc.shared-databases.slick.profile"
    val configMap: mutable.Map[String, String] = mutable.Map()

    dbInfo match {
      case H2(_) =>
        configMap.put(slickProfileConfigKey, H2Profile.getClass.getName)
      case Derby(_) =>
        configMap.put(slickProfileConfigKey, DerbyProfile.getClass.getName)
      case MsSqlServer(_) =>
        configMap.put(slickProfileConfigKey, SQLServerProfile.getClass.getName)
      case MySql(_) =>
        configMap.put(slickProfileConfigKey, MySQLProfile.getClass.getName)
      case Oracle(_) =>
        configMap.put(slickProfileConfigKey, OracleProfile.getClass.getName)
      case PostgreSql(_) =>
        configMap.put(slickProfileConfigKey, PostgresProfile.getClass.getName)

        //for PostgreSql, we have to set table names and column names in lower case
        configMap.put("jdbc-journal.tables.legacy_journal.tableName", "a_job_journal")
        configMap.addAll(getJournalTableColumnConfig("jdbc-journal"))

        configMap.put("jdbc-snapshot-store.tables.legacy_snapshot.tableName", "a_job_snapshot")
        configMap.addAll(getSnapshotTableColumnConfig("jdbc-snapshot-store"))

        configMap.put("sharding-jdbc-journal.tables.legacy_journal.tableName", "a_shard_journal")
        configMap.addAll(getJournalTableColumnConfig("sharding-jdbc-journal"))

        configMap.put("sharding-jdbc-snapshot-store.tables.legacy_snapshot.tableName", "a_shard_snapshot")
        configMap.addAll(getSnapshotTableColumnConfig("sharding-jdbc-snapshot-store"))
      case TestDatabaseInfo(_) =>
        configMap.put(slickProfileConfigKey, H2Profile.getClass.getName)
      case Db2(_) =>
        logger.warn("DB2 Database does not support container based tasks.")
        configMap.put(slickProfileConfigKey, H2Profile.getClass.getName)
      case _ => throw new IllegalStateException("Unsupported database detected")
    }

    ConfigFactory.parseMap(configMap.asJava)
  }

  private def getJournalTableColumnConfig(journalName: String): Map[String, String] = {
    val columnNameConfigKey = s"$journalName.tables.legacy_journal.columnNames"
    Map(
      s"$columnNameConfigKey.ordering" -> "ordering",
      s"$columnNameConfigKey.deleted" -> "deleted",
      s"$columnNameConfigKey.persistenceId" -> "persistence_id",
      s"$columnNameConfigKey.sequenceNumber" -> "sequence_number",
      s"$columnNameConfigKey.created" -> "created",
      s"$columnNameConfigKey.tags" -> "tags",
      s"$columnNameConfigKey.message" -> "message"
    )
  }

  private def getSnapshotTableColumnConfig(snapshotName: String): Map[String, String] = {
    val columnNameConfigKey = s"$snapshotName.tables.legacy_snapshot.columnNames"
    Map(
      s"$columnNameConfigKey.persistenceId" -> "persistence_id",
      s"$columnNameConfigKey.sequenceNumber" -> "sequence_number",
      s"$columnNameConfigKey.created" -> "created",
      s"$columnNameConfigKey.snapshot" -> "snapshot"
    )
  }

  private def actorWithActorSystemOf(actorSystem: ActorSystem,
                                     actorClass: Class[_ <: Actor],
                                     name: String = null,
                                     dispatcher: String = null): ActorRef = {
    if (actorSystem == null) {
      throw new IllegalStateException("Actor system has not created extension yet")
    }
    childActorOf(actorClass, name, dispatcher)(actorSystem)
  }

  def actorOf[T](actorClass: Class[_ <: Actor], name: String, dispatcher: String = null): ManagedActor[T] = {
    logger.debug(s"Creating single managed actor '$name''")
    val holder = createManagedActor[T](name)(as => actorWithActorSystemOf(as, actorClass, name, dispatcher))
    ManagedActorRegistry.put(name, holder)
    holder
  }

  def actorOf[T](props: Props, name: String): ManagedActor[T] = {
    logger.debug(s"Creating single managed actor '$name''")
    val holder = createManagedActor[T](name)(as => as.actorOf(props, name))
    ManagedActorRegistry.put(name, holder)
    holder
  }

  def shardedActorOf[T](actorProps: Props,
                        typeName: String,
                        extractEntityId: ShardRegion.ExtractEntityId,
                        extractShardId: ShardRegion.ExtractShardId,
                        shardingConfig: Option[Config] = None,
                        handOffStopMessage: Any = PoisonPill): ManagedActor[T] = {
    logger.debug(s"Creating sharded single managed actor '$typeName''")
    val holder = createManagedActor[T](typeName)(system => {
      val originalConfig = system.settings.config.getConfig("pekko.cluster.sharding")
      val shardingConfigWithFallback = shardingConfig match {
        case Some(config) => config.withFallback(originalConfig)
        case None => originalConfig
      }
      val shardingSettings = ClusterShardingSettings(shardingConfigWithFallback)
      val sharding = ClusterSharding(system)
      val shardAllocationStrategy = sharding.defaultShardAllocationStrategy(shardingSettings)
      sharding.start(
        typeName = typeName,
        entityProps = actorProps,
        settings = shardingSettings,
        extractEntityId = extractEntityId,
        extractShardId = extractShardId,
        allocationStrategy = shardAllocationStrategy,
        handOffStopMessage = handOffStopMessage
      )
    })
    ManagedActorRegistry.put(typeName, holder)
    holder
  }

  def shardedActorOf[T](actorClazz: Class[_ <: Actor],
                        typeName: String,
                        extractEntityId: ShardRegion.ExtractEntityId,
                        extractShardId: ShardRegion.ExtractShardId,
                        shardingConfig: Option[Config],
                        handOffStopMessage: Any): ManagedActor[T] = {
    val actorProps = props(actorClazz)
    shardedActorOf[T](actorProps, typeName, extractEntityId, extractShardId, shardingConfig, handOffStopMessage)
  }

  def shardedActorOf[T](actorClazz: Class[_ <: Actor],
                        typeName: String,
                        extractEntityId: ShardRegion.ExtractEntityId,
                        extractShardId: ShardRegion.ExtractShardId,
                        shardingConfig: Option[Config]): ManagedActor[T] = {
    val actorProps = props(actorClazz)
    shardedActorOf[T](actorProps, typeName, extractEntityId, extractShardId, shardingConfig, PoisonPill)
  }

  def shardedActorOf[T](actorClazz: Class[_ <: Actor],
                        typeName: String,
                        extractEntityId: ShardRegion.ExtractEntityId,
                        extractShardId: ShardRegion.ExtractShardId): ManagedActor[T] = {
    val actorProps = props(actorClazz)
    shardedActorOf[T](actorProps, typeName, extractEntityId, extractShardId, None, PoisonPill)
  }


  def clusterSingletonActorOf[T](actorClazz: Class[_ <: Actor], actorName: String): ManagedActor[T] = {
    logger.debug(s"Creating cluster singleton managed actor '$actorName'")
    val singletonProps = props(actorClazz)
    clusterSingletonActorOf(singletonProps, actorName)
  }

  def clusterSingletonActorOf[T](actorProps: Props, actorName: String): ManagedActor[T] = {
    logger.debug(s"Creating cluster singleton managed actor '$actorName'")
    val holder = createManagedActor[T](actorName)(system => {
      system.actorOf(ClusterSingletonManager.props(
        singletonProps = actorProps,
        terminationMessage = PoisonPill,
        settings = ClusterSingletonManagerSettings(system)),
        name = actorName)
    })
    ManagedActorRegistry.put(actorName, holder)
    holder
  }

  def clusterSingletonActorProxyOf[T](actorClazz: Class[_ <: Actor], actorName: String): ManagedActor[T] = {
    val singletonProps = props(actorClazz)
    clusterSingletonActorProxyOf(singletonProps, actorName)
  }

  def clusterSingletonActorProxyOf[T](actorProps: Props, actorName: String): ManagedActor[T] = {
    clusterSingletonActorOf(actorProps, actorName)
    val proxyActorName = actorName + "Proxy"
    logger.debug(s"Creating single managed actor '$proxyActorName'")
    val holder = createManagedActor[T](proxyActorName)(as => {
      val props = ClusterSingletonProxy.props(
        singletonManagerPath = s"user/$actorName",
        settings = ClusterSingletonProxySettings(as)
      )
      as.actorOf(props, proxyActorName)
    })
    ManagedActorRegistry.put(proxyActorName, holder)
    holder
  }

  private def createManagedActor[T](actorName: String)(createBlock: ActorCreator): ManagedActor[T] = {
    new ManagedActor[T]() {
      override def resolveTimeout(): Duration = xlrConfig.timeouts.systemInitialization

      override def name(): String = actorName

      override protected def actorCreator: ActorCreator = createBlock

      override def getActorSystem: ActorSystem = ActorSystemHolder.this.getInstance()

      // TODO check if this is what we want or we want to return address/path of actor ref (if it is initialized)
      override def toString: String = actorName

      override def responseTimeout(): Timeout = xlrConfig.timeouts.releaseActionResponse
    }
  }

  private lazy val actorFactory = ActorFactory(applicationContext)

  override def props(actorBeanName: String): Props = actorFactory.props(actorBeanName)

  override def props(actorClass: Class[_ <: Actor]): Props = actorFactory.props(actorClass)
}
