package com.xebialabs.xlrelease.actors.initializer

import com.xebialabs.xlplatform.cluster.ClusterMode._
import com.xebialabs.xlrelease.actors.{ActorServiceHolder, ActorSystemHolder, ArchivePurgingActor}
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{ActorRef, PoisonPill, Props}
import org.apache.pekko.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.core.annotation.Order
import org.springframework.stereotype.Component

trait ArchivePurgingInitializer extends ActorInitializer with Logging {
  def archivePurgingProps(xlrConfig: XlrConfig, actorServiceHolder: ActorServiceHolder): Props = {
    ArchivePurgingActor.props(actorServiceHolder.archivePurgingScheduleService, xlrConfig)
  }
}

@Component
@Order(2)
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredArchivePurgingInitializer @Autowired()(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, actorServiceHolder: ActorServiceHolder)
  extends ArchivePurgingInitializer {

  lazy val archivePurgingActor: ActorRef = systemHolder.actorSystem.actorOf(archivePurgingProps(xlrConfig, actorServiceHolder))

  override def initialize(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug("[MAINTENANCE MODE] Archive purge scheduler actor - DISABLED")
    } else {
      logger.debug("Initializing non-clustered archive purging actor...")
      archivePurgingActor
    }
  }
}

@Component
@Order(2)
@Profile(Array(FULL))
class ClusteredArchivePurgingInitializer @Autowired()(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, actorServiceHolder: ActorServiceHolder)
  extends ArchivePurgingInitializer with Logging {

  lazy val archivePurgingActor: ActorRef = clusteredArchivePurgingActor(archivePurgingProps(xlrConfig, actorServiceHolder))

  override def initialize(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug("[MAINTENANCE MODE] Archive purge scheduler actor - DISABLED")
    } else {
      logger.debug("Initializing clustered archive purging actor...")
      archivePurgingActor
    }
  }

  private[initializer] def clusteredArchivePurgingActor(archivingProps: Props): ActorRef = {
    systemHolder.actorSystem.actorOf(ClusterSingletonManager.props(
      singletonProps = archivingProps,
      terminationMessage = PoisonPill,
      settings = ClusterSingletonManagerSettings(systemHolder.actorSystem)),
      name = ArchivePurgingActor.name)
  }
}

