package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.{ArchivingSettingsManager, XlrConfig}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.events.{ReleaseAbortedEvent, ReleaseCompletedEvent}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.ReleaseRepository
import com.xebialabs.xlrelease.scheduler.{RestartableScheduledExecutorService, XlrExecutors}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import io.micrometer.core.instrument.MeterRegistry
import org.springframework.stereotype.Service

import java.util.concurrent.RejectedExecutionException
import scala.annotation.tailrec

@Service
@EventListener
class PreArchiveService(xlrConfig: XlrConfig,
                        archivingService: ArchivingService,
                        releaseRepository: ReleaseRepository,
                        archivingConfig: ArchivingSettingsManager,
                        meterRegistry: MeterRegistry) extends Logging with XlrServiceLifecycle {

  private lazy val maxThreadCount: Int = xlrConfig.executors.preArchivingExecutor.maxThreadsCount

  private val pool: RestartableScheduledExecutorService = XlrExecutors.newScheduledThreadPool("pre-archiving", maxThreadCount, meterRegistry)

  private var enabled: Boolean = false

  private var catchingUp: Boolean = false

  def isEnabled: Boolean = enabled

  override def getOrder: Int = XlrServiceLifecycleOrder.PRE_ARCHIVE_SERVICE

  def doStart(): Unit = {
    pool.start();
    logger.trace("Setting preArchivingEnabled callback...")
    archivingConfig.subscribeToPreArchivingEnabledChanges { becomesEnabled =>
      this.synchronized {
        if (becomesEnabled) {
          logger.trace("disabled -> enabled")
          enabled = true
          preArchiveInactiveReleasesInBackground()
        } else {
          logger.trace("enabled -> disabled")
          enabled = false
        }
      }
    }

    enabled = archivingConfig.getPreArchivingEnabled
    if (enabled) {
      preArchiveInactiveReleases()
    }
    logger.debug("PreArchiveService started.")
  }

  override def doStop(): Unit = {
    pool.stop()
  }


  @Subscribe
  def preArchiveCompletedRelease(event: ReleaseCompletedEvent): Unit = preArchive(event.release)

  @Subscribe
  def preArchiveAbortedRelease(event: ReleaseAbortedEvent): Unit = preArchive(event.release)

  @Timed
  protected def preArchive(release: Release, checkEnabled: Boolean = true, recoverIfExists: Boolean = true): Unit = {
    if (!checkEnabled || (enabled && !catchingUp)) {
      submit { () =>
        try {
          logger.info(s"Pre-archiving $release")
          archivingService.preArchiveRelease(release)
          logger.debug(s"Pre-archived $release")
        } catch {
          case e: Exception =>
            logger.warn("Something went wrong while pre-archiving", e)
            if (!recoverIfExists) {
              throw e
            }
            // Remove Data from archive db if exist and try again
            if (archivingService.exists(release.getId)) {
              logger.debug(s"Removing $release from Archive Database")
              if (archivingService.deletePreArchiveRelease(release.getId)) {
                logger.debug(s"Pre-archiving again $release")
                preArchive(release, checkEnabled, recoverIfExists = false)
              } else {
                logger.debug(s"Removing failed $release from Archive Database")
              }
            } else {
              logger.debug(s"Release $release does not exist in Archive Database", e)
              throw e
            }
        }
      }
    } else {
      logger.debug(s"Pre-archiving is disabled")
    }
  }

  private def preArchiveInactiveReleasesInBackground(): Unit = synchronized {
    if (!catchingUp) {
      catchingUp = true
      submit(() => {
        preArchiveInactiveReleases()
        catchingUp = false
      })
    }
  }

  @tailrec
  private def preArchiveInactiveReleases(page: Int = 0): Unit = {
    val pageSize = archivingConfig.getSearchPageSize
    logger.debug(s"Fetching page $page ($pageSize releases) for pre-archiving...")
    val found = releaseRepository.findPreArchivableReleases(page, pageSize)
    found.filter(archivingService.shouldArchive).foreach(preArchive(_, checkEnabled = false))

    if (enabled && found.size == pageSize) {
      preArchiveInactiveReleases(page + 1)
    }
  }

  private def submit[U](job: () => U): Unit = {
    if (pool.isActive) {
      try {
        pool.submit(runnable(job))
      } catch {
        case e: RejectedExecutionException => logger.warn(s"Unable to submit a job to ${pool.name()}", e)
      }
    } else {
      logger.warn(s"Pool ${pool.name()} is ${pool.state}. Cannot process job.")
    }
  }

  private def runnable[U](body: () => U): Runnable = () => body()
}
