package com.xebialabs.xlrelease.scheduler

import com.xebialabs.xlrelease.domain.management.ServiceState
import com.xebialabs.xlrelease.domain.management.ServiceState.ServiceState
import com.xebialabs.xlrelease.scheduler.RestartableExecutorService.{DEFAULT_SHUTDOWN_TIMEOUT, NOT_AVAILABLE_MSG}
import grizzled.slf4j.Logging

import java.util
import java.util.Collections
import java.util.concurrent._

trait RestartableExecutorService extends ExecutorService {
  def state: ServiceState

  def start(): Unit

  def stop(): Unit

  def enable(): Unit = {
    start()
  }

  def disable(): Unit = {
    stop()
  }

  def isActive: Boolean = this.state == ServiceState.Running || this.state == ServiceState.Stopping

  def name(): String

}

class RestartableExecutor[R <: ExecutorService](name: String,
                                                executorFactory: () => R,
                                                shutdownTimeout: Long = DEFAULT_SHUTDOWN_TIMEOUT)
  extends RestartableExecutorService
    with CoordinatedShutdown
    with Logging {

  @volatile
  private var _state: ServiceState = ServiceState.Stopped
  private var _original: R = _

  protected def original: Option[R] = {
    if (this.isActive) {
      Some(_original)
    } else {
      None
    }
  }

  override def initShutdown(): Unit = {
    this._state = ServiceState.Stopping
    original.foreach(_.shutdown())
  }

  override def shutdownNow(): util.List[Runnable] = original.map(_.shutdownNow()).getOrElse(Collections.emptyList())

  override def shutdown(): Unit = {
    this.coordinatedShutdown()
    this._state = ServiceState.Stopped
  }

  override def start(): Unit = {
    synchronized {
      logger.info(s"Starting $name restartable executor")
      if (!isActive) {
        this._original = executorFactory()
      }
      this._state = ServiceState.Running
    }
  }

  override def stop(): Unit = {
    synchronized {
      shutdown()
    }
  }

  override def name(): String = s"$name"

  override def shutdownTimeout(): Long = shutdownTimeout

  def state: ServiceState = _state

  // Delegated executor service methods
  override def isShutdown: Boolean = original.forall(_.isShutdown)

  override def isTerminated: Boolean = original.forall(_.isTerminated)

  override def awaitTermination(timeout: Long, unit: TimeUnit): Boolean = original.forall(_.awaitTermination(timeout, unit))

  override def submit[T](task: Callable[T]): Future[T] = {
    original.map(_.submit(task)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def submit[T](task: Runnable, result: T): Future[T] = {
    original.map(_.submit(task, result)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def submit(task: Runnable): Future[_] = {
    original.map(_.submit(task)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAll[T](tasks: util.Collection[_ <: Callable[T]]): util.List[Future[T]] = {
    original.map(_.invokeAll(tasks)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAll[T](tasks: util.Collection[_ <: Callable[T]], timeout: Long, unit: TimeUnit): util.List[Future[T]] = {
    original.map(_.invokeAll(tasks, timeout, unit)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAny[T](tasks: util.Collection[_ <: Callable[T]]): T = {
    original.map(_.invokeAny(tasks)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def invokeAny[T](tasks: util.Collection[_ <: Callable[T]], timeout: Long, unit: TimeUnit): T = {
    original.map(_.invokeAny(tasks, timeout, unit)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }

  override def execute(command: Runnable): Unit = {
    original.map(_.execute(command)).getOrElse(throw new RejectedExecutionException(NOT_AVAILABLE_MSG))
  }
}

object RestartableExecutorService {
  val NOT_AVAILABLE_MSG = "Executor not available"
  val DEFAULT_SHUTDOWN_TIMEOUT = 10
}
