package com.xebialabs.xlrelease.metrics

import com.codahale.metrics.MetricRegistry
import com.codahale.metrics.jmx.JmxReporter
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.metrics.MetricsConfiguration.METRICS_DOMAIN
import com.zaxxer.hikari.metrics.MetricsTrackerFactory
import com.zaxxer.hikari.metrics.micrometer.MicrometerMetricsTrackerFactory
import io.micrometer.core.aop.{MeterTagAnnotationHandler, TimedAspect}
import io.micrometer.core.instrument._
import io.micrometer.core.instrument.binder.jvm._
import io.micrometer.core.instrument.binder.logging.LogbackMetrics
import io.micrometer.core.instrument.binder.system.ProcessorMetrics
import io.micrometer.core.instrument.config.NamingConvention
import io.micrometer.core.instrument.util.HierarchicalNameMapper
import io.micrometer.jmx.{JmxConfig, JmxMeterRegistry}
import org.springframework.beans.factory.ObjectProvider
import org.springframework.context.annotation.{Bean, Configuration}

import java.net.InetAddress
import java.util.concurrent.TimeUnit
import scala.util.Try

object MetricsConfiguration {
  val METRICS_DOMAIN = "com.xebialabs.xlrelease.metrics"

  val TAG_EXECUTOR = Tag.of("type", "executor")
}

@Configuration(proxyBeanMethods = false)
class MetricsConfiguration {

  @Bean
  def jmxConfig: JmxConfig = (key: String) => null

  @Bean
  def metricClock: Clock = Clock.SYSTEM

  @Bean
  def timedAspect(registry: MeterRegistry, meterTagAnnotationHandler: ObjectProvider[MeterTagAnnotationHandler]) = {
    val timedAspect = new TimedAspect(registry)
    meterTagAnnotationHandler.ifAvailable(timedAspect.setMeterTagAnnotationHandler)
    timedAspect
  }

  @Bean
  def jmxMeterRegistry(config: JmxConfig,
                       clock: Clock,
                       serverConfiguration: ServerConfiguration): JmxMeterRegistry = {
    val metricRegistry = new MetricRegistry()
    val jmxReporter = JmxReporter.forRegistry(metricRegistry)
      .inDomain(METRICS_DOMAIN)
      .convertRatesTo(TimeUnit.SECONDS)
      .createsObjectNamesWith(XlrObjectNameFactory)
      .build

    val instanceName = getUniqueReleaseInstanceName(serverConfiguration)
    val registry = new JmxMeterRegistry(config, clock, hierarchicalNameMapper, metricRegistry, jmxReporter)
    registry.config()
      .namingConvention(NamingConvention.identity)
      .commonTags(Tags.of("xlrelease-instance", instanceName))

    if (XlrConfig.getInstance.metrics.enabled) {
      bindSystemMetrics(registry)
    } else {
      jmxReporter.stop()
    }

    registry
  }

  @Bean
  def metricsTrackerFactory(meterRegistry: MeterRegistry): MetricsTrackerFactory = {
    val metricsTrackerFactory = new MicrometerMetricsTrackerFactory(meterRegistry)
    metricsTrackerFactory
  }

  private def bindSystemMetrics(registry: MeterRegistry): Unit = {
    new LogbackMetrics().bindTo(registry)
    new ClassLoaderMetrics().bindTo(registry)
    new JvmMemoryMetrics().bindTo(registry)
    new JvmGcMetrics().bindTo(registry)
    new ProcessorMetrics().bindTo(registry)
    new JvmThreadMetrics().bindTo(registry)
  }

  private def hierarchicalNameMapper(id: Meter.Id, convention: NamingConvention) = {
    import scala.jdk.CollectionConverters._
    // remove common tags
    val newTags = id.getTags.asScala.filterNot(0 == _.compareTo(Tag.of("xlrelease-instance", ""))).asJava
    val idWithoutCommonTags = id.replaceTags(newTags)
    val tagType = Option(idWithoutCommonTags.getTag("type"))
    val tagName = Option(idWithoutCommonTags.getTag("name"))

    (tagType, tagName) match {
      case (Some(beanType), Some(beanName)) =>
        s"$beanType.$beanName.${idWithoutCommonTags.getName}"
      case _ =>
        HierarchicalNameMapper.DEFAULT.toHierarchicalName(idWithoutCommonTags, convention)
    }
  }

  private def getUniqueReleaseInstanceName(serverConfiguration: ServerConfiguration): String = {
    val serverHost = Try(InetAddress.getLocalHost.getHostName).getOrElse(serverConfiguration.getHttpBindAddress)
    val serverPort = serverConfiguration.getHttpPort
    s"${serverHost}_$serverPort"
  }

}
