package com.xebialabs.xlrelease.scheduler.service

import com.xebialabs.xlrelease.actors.cluster.XlrCluster.ClusterMemberOps
import com.xebialabs.xlrelease.actors.{ActorSystemHolder, ReleaseActorService}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.{JobQueue, JobRecoveryLogic, JobStatus}
import org.apache.pekko.actor.ActorSystem
import org.apache.pekko.cluster.Cluster
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util


@Service
class JobCleanupService @Autowired()(val jobQueue: JobQueue,
                                     val jobService: JobService,
                                     val actorSystemHolder: ActorSystemHolder,
                                     val releaseActorService: ReleaseActorService) extends JobRecoveryLogic {

  @volatile
  private var isStarted = false

  def start(): Unit = {
    isStarted = true
  }

  def stop(): Unit = {
    isStarted = false
  }

  lazy val actorSystem: ActorSystem = actorSystemHolder.getInstance()

  def detectAbandonedTasks(): Unit = {
    // this is 2nd entry point periodically executed from ScheduledTaskRecoveryJob
    if (isStarted) {
      val taskNodeIds = jobService.findDistinctNodeIds().filterNot(_ == null)
      logger.debug(s"distinct nodeIds of tasks: ${taskNodeIds.mkString(", ")}")
      val aliveNodeIds = Cluster(actorSystem).state.members
        .filter(_.isAlive)
        .map(_.address)
        .map(convertAddressToNodeId)
      logger.debug(s"alive nodeIds: ${aliveNodeIds.mkString(", ")}")
      val deadNodes = taskNodeIds.diff(aliveNodeIds)
      deadNodes.foreach(nodeId => {
        logger.warn(s"Discovered dead node $nodeId, failing/recovering tasks assigned to that node")
        recoverJobs(nodeId)
      })
    } else {
      logger.debug("skipping detectAbandonedTasks because service is not completely initialized yet")
    }
  }

  def detectStaleReservedJobs(): Unit = {
    // if job is in RESERVED status for period longer than configured we should re-queue it
    val jobFilters = new JobFilters()
    jobFilters.status = util.Arrays.asList(JobStatus.RESERVED.name())
    // TODO do we have to make query more efficient (filter in DB?)
    val taskJobs = jobService.findAllTaskJobs(jobFilters, Pageable.unpaged(), ignoreUnknownType = true)
    val staleReservedJobDetectionPeriodInSeconds = XlrConfig.getInstance.timeouts.staleReservedJobDetectionPeriod.toSeconds
    val earliestAllowedJobInstant = Instant.now().minus(staleReservedJobDetectionPeriodInSeconds, ChronoUnit.SECONDS)
    for {
      lateJob <- taskJobs.filter(j => j.reservationTime == null || j.reservationTime.isBefore(earliestAllowedJobInstant))
      updatedJobRow <- jobService.updateJobNodeAndStatus(lateJob, null, JobStatus.QUEUED)
      updatedJob <- {
        lateJob.version = updatedJobRow.version
        lateJob.node = updatedJobRow.node
        lateJob.status = updatedJobRow.status
        Option(lateJob)
      }
    } jobQueue.submitExisting(updatedJob)
  }
}
