package com.xebialabs.xlrelease.actors.cluster

import com.xebialabs.xlplatform.cluster.ClusterMode
import com.xebialabs.xlplatform.cluster.ClusterMode.{Full, Standalone}
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.cluster.XlrCluster.{ClusterManagerConstants, LEGACY_DOWNING_PROVIDERS, ServiceDiscoveryConstants}
import com.xebialabs.xlrelease.actors.initializer.ActorSystemInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.repository.ClusterMembersRepository
import com.xebialabs.xlrelease.support.pekko.spring.SpringExtension
import org.springframework.context.annotation.{Bean, Configuration}

@Configuration
class XlrClusterConfiguration(xlrConfig: XlrConfig,
                              actorSystemHolder: ActorSystemHolder,
                              actorsInitializer: ActorSystemInitializer,
                              clusterMembersRepository: ClusterMembersRepository,
                              springExtension: SpringExtension) {

  @Bean
  def xlrActorSystemBootstrap: XlrActorSystemBootstrap = {
    val clusterMode = XlrConfig.getInstance.clusterMode
    if (clusterMode == ClusterMode.Standalone) {
      XlrStandaloneNodeBootstrap(actorSystemHolder, actorsInitializer)
    } else {
      val downingProvider = xlrConfig.pekko.pekkoCluster.downingProviderClass
      xlrConfig.clusterManagement.clusterManager match {
        case ClusterManagerConstants.PEKKO_NATIVE =>
          if (LEGACY_DOWNING_PROVIDERS.contains(downingProvider)) {
            throw new IllegalStateException(
              s"Downing provider $downingProvider is not supported with xl.cluster.manager=\"${ClusterManagerConstants.PEKKO_NATIVE}\""
            )
          }
          XlrPekkoNativeClusterBootstrap(xlrConfig, actorSystemHolder, xlrClusterMembersService)
        case ClusterManagerConstants.XLR_LEGACY =>
          if (!LEGACY_DOWNING_PROVIDERS.contains(downingProvider)) {
            throw new IllegalStateException(
              s"Downing provider $downingProvider is not supported with xl.cluster.manager=\"${ClusterManagerConstants.XLR_LEGACY}\""
            )
          }
          XlrLegacyClusterBootstrap(xlrConfig, actorSystemHolder, actorsInitializer)
        case _ =>
          throw new IllegalStateException(s"Unsupported cluster manager \"${xlrConfig.clusterManagement.clusterManager}\"")
      }
    }
  }

  @Bean
  def clusterStatisticsProvider(): ClusterStatisticsProvider = {
    val clusterManager = xlrConfig.clusterManagement.clusterManager
    val clusterMode = xlrConfig.clusterMode
    clusterMode match {
      case Full =>
        clusterManager match {
          case ClusterManagerConstants.PEKKO_NATIVE =>
            XlrPekkoNativeClusterStatisticsProvider(actorSystemHolder, xlrClusterMembersService)
          case ClusterManagerConstants.XLR_LEGACY =>
            XlrLegacyClusterStatisticsProvider(xlrConfig, actorSystemHolder)
          case _ =>
            throw new IllegalStateException(s"Unsupported cluster manager \"${xlrConfig.clusterManagement.clusterManager}\"")
        }
      case Standalone =>
        StandaloneClusterStatisticsProvider(actorSystemHolder)
      case _ =>
        throw new IllegalStateException(s"Unsupported cluster mode \"${xlrConfig.clusterMode}\"")
    }

  }

  @Bean
  def xlrClusterMembersService: ClusterMembersService = {
    val clusterMode = XlrConfig.getInstance.clusterMode
    if (clusterMode == ClusterMode.Full &&
      xlrConfig.clusterManagement.clusterManager == ClusterManagerConstants.PEKKO_NATIVE &&
      xlrConfig.pekko.discovery.method == ServiceDiscoveryConstants.JDBC
    ) {
      JdbcClusterMembersService(xlrConfig, clusterMembersRepository, XlrClusterInformationImpl(actorSystemHolder), springExtension)
    } else {
      NoopClusterMembersService
    }
  }
}
