package com.xebialabs.xlrelease.actors.cluster

import com.xebialabs.xlplatform.cluster.ClusterMode.{FULL, Full, HotStandby}
import com.xebialabs.xlplatform.cluster.XlCluster
import com.xebialabs.xlplatform.cluster.full.downing.LeaderAutoDowningActor.NODE_DOWNED_EXIT_CODE
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.initializer.ActorSystemInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging
import org.apache.pekko.management.cluster.bootstrap.ClusterBootstrap
import org.apache.pekko.management.scaladsl.PekkoManagement
import org.springframework.boot.SpringApplication

import scala.concurrent.ExecutionContextExecutor
import scala.util.{Failure, Success}

trait XlrActorSystemBootstrap {

  def actorSystemHolder: ActorSystemHolder

  def boot(): Unit

  def stop(): Unit = {
    actorSystemHolder.shutdown()
  }
}


case class XlrLegacyClusterBootstrap(xlrConfig: XlrConfig,
                                     actorSystemHolder: ActorSystemHolder,
                                     actorsInitializer: ActorSystemInitializer) extends XlrActorSystemBootstrap {


  override def boot(): Unit = {
    xlrConfig.cluster.mode match {
      case HotStandby =>
        throw new UnsupportedOperationException(
          s"HotStandby cluster mode is no longer supported, switch to $FULL if you want to boot in the cluster mode.")
      case Full =>
        val mgmtActorSystem = actorSystemHolder.actorSystem
        XlCluster.init(xlrConfig.cluster)
        XlCluster(mgmtActorSystem).start(() => initialize())
      case _ => throw new UnsupportedOperationException(s"Unsupported cluster mode \"${xlrConfig.cluster.mode}\" for legacy cluster manager")
    }
  }

  private def initialize(): Unit = actorsInitializer.initialize()
}

case class XlrPekkoNativeClusterBootstrap(xlrConfig: XlrConfig,
                                          actorSystemHolder: ActorSystemHolder,
                                          clusterMembersService: ClusterMembersService
                                         ) extends XlrActorSystemBootstrap with Logging with ScalaSpringAwareBean {
  override def boot(): Unit = {
    xlrConfig.cluster.mode match {
      case Full =>
        val system = actorSystemHolder.actorSystem
        implicit val ec: ExecutionContextExecutor = system.dispatcher
        val managementFuture = PekkoManagement(system).start()
        ClusterBootstrap(system).start()
        managementFuture.onComplete {
          case Success(_) => initialize()
          case Failure(e) =>
            logger.error("Pekko management failed to start", e)
            SpringApplication.exit(applicationContext, () => NODE_DOWNED_EXIT_CODE)
        }
      case _ => throw new UnsupportedOperationException(s"Unsupported cluster mode \"${xlrConfig.cluster.mode}\" for Pekko cluster manager")
    }
  }

  private def initialize(): Unit = {
    try {
      clusterMembersService.registerSelf()
      ClusterEventsListenerActor.start()
    } catch {
      case e: Exception =>
        logger.error("Failed to initialize cluster", e)
        SpringApplication.exit(applicationContext, () => NODE_DOWNED_EXIT_CODE)
    }
  }
}

case class XlrStandaloneNodeBootstrap(actorSystemHolder: ActorSystemHolder,
                                      actorsInitializer: ActorSystemInitializer) extends XlrActorSystemBootstrap {

  override def boot(): Unit = {
    actorsInitializer.initialize()
  }
}

