package org.springframework.data.domain;

import java.io.Serial;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

/**
 * This is a copy of {@link org.springframework.data.domain.PageImpl} with a change in constructor.
 * This keeps total value as it is instead of modifying it.
 */
public class ReleasePageImpl<T> extends Chunk<T> implements Page<T> {
    @Serial
    private static final long serialVersionUID = -6969139863724462053L;
    private final long total;

    public ReleasePageImpl(List<T> content, Pageable pageable, long total) {
        super(content, pageable);
        this.total = total;
    }

    public ReleasePageImpl(List<T> content) {
        this(content, Pageable.unpaged(), content.size());
    }

    @Override
    public int getTotalPages() {
        return this.getSize() == 0 ? 1 : (int) Math.ceil((double) this.total / (double) this.getSize());
    }

    @Override
    public long getTotalElements() {
        return this.total;
    }

    @Override
    public boolean hasNext() {
        return this.getNumber() + 1 < this.getTotalPages();
    }

    @Override
    public boolean isLast() {
        return !this.hasNext();
    }

    @Override
    public <U> Page<U> map(Function<? super T, ? extends U> converter) {
        return new ReleasePageImpl<>(this.getConvertedContent(converter), this.getPageable(), this.total);
    }

    @Override
    public String toString() {
        String contentType = "UNKNOWN";
        List<T> content = this.getContent();
        if (!content.isEmpty() && content.get(0) != null) {
            contentType = content.get(0).getClass().getName();
        }

        return String.format("Page %s of %d containing %s instances", this.getNumber() + 1, this.getTotalPages(), contentType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        ReleasePageImpl<?> that = (ReleasePageImpl<?>) o;
        return total == that.total;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), total);
    }
}
