package com.xebialabs.xlrelease.actors.cluster

import com.xebialabs.xlrelease.actors.cluster.ClusterMemberHeartbeatActor.{Start, Stop}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.ClusterMembersRepository
import com.xebialabs.xlrelease.support.pekko.spring.SpringExtension
import grizzled.slf4j.Logging

import java.time.Instant


trait ClusterMembersService {
  def registerSelf(): Unit

  def leave(): Unit

  def listActiveMembers(): Seq[ClusterMember]

  def cleanupExpiredMembers(): Unit
}

@IsTransactional
case class JdbcClusterMembersService(xlrConfig: XlrConfig,
                                     clusterMembersRepository: ClusterMembersRepository,
                                     clusterInformation: XlrClusterInformation,
                                     springExtension: SpringExtension
                                    ) extends ClusterMembersService with Logging {


  private lazy val livenessActor = springExtension.actorOf(classOf[ClusterMemberHeartbeatActor], "clusterMemberHeartbeatActor")

  override def registerSelf(): Unit = {
    val expiresAt = calculateNewExpirationTime()
    if (clusterMembersRepository.exits(clusterInformation.myself)) {
      clusterMembersRepository.updateTtl(clusterInformation.myself, expiresAt)
    } else {
      clusterMembersRepository.create(clusterInformation.myself, expiresAt)
    }
    livenessActor ! Start()
  }

  override def leave(): Unit = {
    clusterMembersRepository.delete(clusterInformation.myself)
    livenessActor ! Stop()
  }

  override def listActiveMembers(): Seq[ClusterMember] = {
    val expiresAt = Instant.now()
    clusterMembersRepository.listActiveMembers(expiresAt)
  }

  private def calculateNewExpirationTime(): Instant = {
    val ttl = xlrConfig.pekko.discovery.jdbcDiscovery.ttl
    Instant.now().plusMillis(ttl.toMillis)
  }

  override def cleanupExpiredMembers(): Unit = {
    val twiceTheTtl = xlrConfig.pekko.discovery.jdbcDiscovery.ttl * 2
    val expiresAt = Instant.now().minusMillis(twiceTheTtl.toMillis)
    clusterMembersRepository.deleteExpiredMembers(expiresAt)
  }
}

case object NoopClusterMembersService extends ClusterMembersService {

  override def registerSelf(): Unit = {
    //Do nothing
  }

  override def leave(): Unit = {
    //Do nothing
  }

  override def listActiveMembers(): List[ClusterMember] = {
    //Do nothing
    List()
  }

  override def cleanupExpiredMembers(): Unit = {
    //
  }
}


case class ClusterMember(datacenter: String, protocol: String, system: String, host: String, managementPort: Int, port: Int)
