package com.xebialabs.xlrelease.support.report

import com.xebialabs.xlplatform.support.report.ReportDataProvider
import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.support.report.repository.DataStatisticsRepository
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.beans.BeanProperty
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.math.BigDecimal.RoundingMode

@Component
class DataStatisticsProvider @Autowired()(val repository: DataStatisticsRepository,
                                          val archivingSettingsManager: ArchivingSettingsManager) extends ReportDataProvider {

  val name: String = "data"

  //noinspection ScalaStyle
  override def collectStatistics: Map[String, Any] = {
    Map[String, Any]("data" -> mutable.Map[String, Any](
      "releases" -> collectReleaseUsageStatistics,
      "tasks" -> collectTaskUsageStatistics,
      "templates" -> collectTemplateUsageStatistics,
      "triggers" -> collectTriggerUsageStatistics,
      "folders" -> collectFolderUsageStatistics,
      "variables" -> collectVariableUsageStatistics,
      "users" -> collectUserUsageStatistics,
      "dashboards" -> collectDashboardUsageStatistics,
      "groups" -> collectGroupUsageStatistics,
      "deliveries" -> collectDeliveryUsageStatistics,
      "configurations" -> collectConfigurationUsageStatistics,
      "applications" -> collectApplicationUsageStatistics,
      "environments" -> collectEnvironmentsUsageStatistics,
      "calendarEntries" -> collectCalendarUsageStatistics,
      "archive" -> collectArchiveUsageStatistics,
      "report" -> collectReportUsageStatistics,
      "workflows" -> collectWorkflowUsageStatistics
    ).asJava)
  }

  def collectReleaseUsageStatistics: ReleaseUsageStatistics = {
    val statistics = new ReleaseUsageStatistics
    val releaseCountByStatus = repository.countReleasesByStatus().map(entry => entry.status -> entry.count).toMap
    statistics.plannedReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.PLANNED.value(), 0)
    statistics.inProgressReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.IN_PROGRESS.value(), 0)
    statistics.pausedReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.PAUSED.value(), 0)
    statistics.failingReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.FAILING.value(), 0)
    statistics.failedReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.FAILED.value(), 0)
    statistics.completedReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.COMPLETED.value(), 0)
    statistics.abortedReleasesTotal = releaseCountByStatus.getOrElse(ReleaseStatus.ABORTED.value(), 0)
    statistics.releasesLast24HoursTotal = repository.countReleasesInLastDays(1)
    statistics.releasesLast30DaysTotal = repository.countReleasesInLastDays(30)
    statistics.completionRateLastMonth = repository.calculateCompletionRateLastMonth()
    statistics.averageReleasesPerDayLastYear = repository.averageReleasesInLastDays(365)
    statistics.averageReleasesPerMonthLastYear = repository.averageReleasesPerMonthInLastYear()
    statistics.releasesWithDependencies = repository.countReleasesWithDependencies()
    statistics.dependenciesTotal = repository.countDependencies()
    statistics.averageDependenciesPerRelease = repository.averageDependenciesPerRelease()
    statistics.maxDependenciesInRelease = repository.findMaxDependenciesInRelease()
    statistics.releaseTagsTotal = repository.countDistinctReleaseTags()
    statistics.averagePhasesPerRelease = repository.averagePhasesPerRelease()
    statistics.maxPhasesInRelease = repository.findMaxPhasesInRelease()
    statistics.maxSecurableCiUidValue = repository.maxSecurableCiUidValue()
    statistics
  }

  def collectTaskUsageStatistics: TaskUsageStatistics = {
    val statistics = new TaskUsageStatistics
    statistics.plannedTasksTotal = repository.countPlannedTasks()
    val taskCountByStatus = repository.countTasksByStatuses().map(entry => entry.status -> entry.count).toMap
    statistics.pendingTasksTotal = taskCountByStatus.getOrElse(TaskStatus.PENDING.value(), 0)
    statistics.inProgressTasksTotal = taskCountByStatus.getOrElse(TaskStatus.IN_PROGRESS.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.WAITING_FOR_INPUT.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.PRECONDITION_IN_PROGRESS.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.FAILURE_HANDLER_IN_PROGRESS.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.FACET_CHECK_IN_PROGRESS.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.ABORT_SCRIPT_IN_PROGRESS.value(), 0)
    statistics.failedTasksTotal = taskCountByStatus.getOrElse(TaskStatus.FAILED.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.FAILING.value(), 0)
    statistics.skippedTasksTotal = taskCountByStatus.getOrElse(TaskStatus.SKIPPED.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.SKIPPED_IN_ADVANCE.value(), 0)
    statistics.completedTasksTotal = taskCountByStatus.getOrElse(TaskStatus.COMPLETED.value(), 0) +
      taskCountByStatus.getOrElse(TaskStatus.COMPLETED_IN_ADVANCE.value(), 0)
    statistics.abortedTasksTotal = taskCountByStatus.getOrElse(TaskStatus.ABORTED.value(), 0)
    statistics.templateTasksTotal = taskCountByStatus.getOrElse(TaskStatus.PLANNED.value(), 0) - statistics.plannedTasksTotal
    statistics.maxTasksInRelease = repository.findMaxTasksInRelease()
    statistics.averageTasksPerRelease = repository.averageTasksPerRelease()
    statistics.top25TaskTypes = repository.countTopTaskTypeOccurrences(25).asJava
    statistics.automationPercentage = calculateAutomationPercentage(repository.countAutomatedTasks(), repository.countTasks())
    statistics.taskTagsTotal = repository.countDistinctTasksTags()
    statistics.taskBackupsTotal = repository.countTaskBackups()
    statistics.commentsTotal = repository.countTaskComments()
    statistics.attachmentsTotal = repository.countAttachments()
    statistics.maxTasksCiUidValue = repository.maxTasksCiUidValue()
    statistics
  }

  def collectUserUsageStatistics: UserUsageStatistics = {
    val statistics = new UserUsageStatistics
    statistics.internalUsersTotal = repository.countUsers()
    statistics.activeUsersLastDay = repository.countActiveUsersInLastDays(1)
    statistics.activeUsersLast30Days = repository.countActiveUsersInLastDays(30)
    statistics.activeUsersLast90Days = repository.countActiveUsersInLastDays(90)
    statistics.globalRolesTotal = repository.countGlobalRoles()
    statistics.usersTotal = repository.countUserProfiles()
    statistics.externalUsersTotal = statistics.usersTotal - statistics.internalUsersTotal
    statistics.enabledUsersTotal = repository.countEnabledUsers()
    statistics.teamsTotal = repository.countTeams()
    statistics.roleRolesTotal = repository.countRoleRoles()
    statistics.rolePrincipalsTotal = repository.countRolePrincipals()
    statistics.rolePermissionsTotal = repository.countRolePermissions()
    statistics
  }

  def collectTemplateUsageStatistics: TemplateUsageStatistics = {
    val statistics = new TemplateUsageStatistics
    statistics.templatesTotal = repository.countTemplates()
    statistics.templateRevisionsTotal = repository.countTemplateRevisions()
    statistics
  }

  def collectTriggerUsageStatistics: TriggerUsageStatistics = {
    val statistics = new TriggerUsageStatistics
    statistics.triggersTotal = repository.countTriggers()
    statistics.activeTriggersTotal = repository.countActiveTriggers()
    statistics
  }

  def collectFolderUsageStatistics: FolderUsageStatistics = {
    val statistics = new FolderUsageStatistics
    statistics.foldersTotal = repository.countFolders()
    statistics.maxDepthOfFolder = repository.maxFolderDepth()
    statistics
  }

  def collectVariableUsageStatistics: VariableUsageStatistics = {
    val statistics = new VariableUsageStatistics
    val folderVariableCount = repository.countFolderVariables()
    statistics.globalVariablesTotal = repository.countGlobalVariables()
    statistics.folderVariablesTotal = folderVariableCount
    statistics.averageVariablesPerFolder = avg(folderVariableCount, repository.countFolders())
    statistics
  }

  def collectDashboardUsageStatistics: DashboardUsageStatistics = {
    val statistics = new DashboardUsageStatistics
    statistics.globalDashboardsTotal = repository.countGlobalDashboards()
    statistics.folderDashboardsTotal = repository.countFolderDashboards()
    statistics
  }

  def collectGroupUsageStatistics: ReleaseGroupUsageStatistics = {
    val statistics = new ReleaseGroupUsageStatistics
    statistics.groupsTotal = repository.countReleaseGroups()
    statistics
  }

  def collectDeliveryUsageStatistics: DeliveryUsageStatistics = {
    val statistics = new DeliveryUsageStatistics
    statistics.patternsTotal = repository.countDeliveryPatterns()
    statistics.deliveriesTotal = repository.countDeliveries()
    statistics
  }

  def collectArchiveUsageStatistics: ArchiveUsageStatistics = {
    val statistics = new ArchiveUsageStatistics
    statistics.archivingEnabled = archivingSettingsManager.getEnabled
    statistics.archivingAge = archivingSettingsManager.getReleaseAgeToDeleteFromJcr / 24
    statistics.archivedReleasesTotal = repository.countArchivedReleases()
    statistics.prearchivedReleasesTotal = repository.countPreArchivedReleases()
    statistics.oldestArchivedRelease = Option(repository.oldestArchivedRelease()).getOrElse("None")
    statistics.archivedCompletedReleasesTotal = repository.countArchivedReleasesByState(ReleaseStatus.COMPLETED.value())
    statistics.archivedAbortedReleasesTotal = repository.countArchivedReleasesByState(ReleaseStatus.ABORTED.value())
    statistics.archivedPhasesTotal = repository.countArchivedPhases()
    statistics.archivedTasksTotal = repository.countArchivedTasks()
    statistics.archivedAttachmentsTotal = repository.countArchivedAttachments()
    statistics.archivedTagsTotal = repository.countArchivedTags()
    statistics.archivedRolesTotal = repository.countArchivedRoles()
    statistics.archivedMembersTotal = repository.countArchivedMembers()
    statistics.workflowsArchivedCompleted = repository.countArchivedReleasesByState(ReleaseStatus.COMPLETED.value(), ReleaseKind.WORKFLOW)
    statistics.workflowsArchivedAborted = repository.countArchivedReleasesByState(ReleaseStatus.ABORTED.value(), ReleaseKind.WORKFLOW)
    statistics.workflowsArchivedTotal = repository.countArchivedReleases(ReleaseKind.WORKFLOW)
    statistics.workflowsPrearchivedTotal = repository.countPreArchivedReleases(ReleaseKind.WORKFLOW)
    statistics.maxArchivedTasksUidValue = repository.maxArchivedTasksUidValue()

    statistics
  }

  def collectConfigurationUsageStatistics: ConfigurationUsageStatistics = {
    val statistics = new ConfigurationUsageStatistics
    statistics.globalConnectionsTotal = repository.countGlobalConfigurations()
    statistics.folderConnectionsTotal = repository.countFolderConfigurations()
    statistics
  }

  def collectApplicationUsageStatistics: ApplicationUsageStatistics = {
    val statistics = new ApplicationUsageStatistics
    statistics.applicationsTotal = repository.countApplications()
    statistics
  }

  def collectEnvironmentsUsageStatistics: EnvironmentUsageStatistics = {
    val statistics = new EnvironmentUsageStatistics
    statistics.environmentsTotal = repository.countEnvironments()
    statistics
  }

  def collectCalendarUsageStatistics: CalendarUsageStatistics = {
    val statistics = new CalendarUsageStatistics
    statistics.calendarEntriesTotal = repository.countCalendarEntries()
    statistics
  }

  def collectReportUsageStatistics: ReportUsageStatistics = {
    val statistics = new ReportUsageStatistics
    statistics.activityLogsTotal = repository.countActivityLogs()
    statistics.attributesTotal = repository.countAttributes()
    statistics.deploymentsTotal = repository.countDeployments()
    statistics.deploymentHistoriesTotal = repository.countDeploymentHistories()
    statistics.deploymentRecordsTotal = repository.countDeploymentRecords()
    statistics.itsmRecordsTotal = repository.countItsmRecords()
    statistics.complianceRecordsTotal = repository.countComplianceRecords()
    statistics.buildRecordsTotal = repository.countBuildRecords()
    statistics.planRecordsTotal = repository.countPlanRecords()
    statistics.archivedDeploymentRecordsTotal = repository.countArchivedDeploymentRecords()
    statistics.archivedItsmRecordsTotal = repository.countArchivedItsmRecords()
    statistics.archivedComplianceRecordsTotal = repository.countArchivedComplianceRecords()
    statistics.archivedBuildRecordsTotal = repository.countArchivedBuildRecords()
    statistics.archivedPlanRecordsTotal = repository.countArchivedPlanRecords()
    statistics.permissionSnapshotsTotal = repository.countPermissionSnapshots()
    statistics.riskAssessmentsTotal = repository.countRiskAssessments()
    statistics.maxActivityLogIdValue = repository.maxActivityLogIdValue()
    statistics
  }

  private def calculateAutomationPercentage(automatedTasks: Int, totalTasks: Int): Float = {
    if (totalTasks > 0) (BigDecimal(automatedTasks) / BigDecimal(totalTasks)).setScale(2, RoundingMode.HALF_UP).floatValue else 0F
  }

  def collectWorkflowUsageStatistics: WorkflowUsageStatistics = {
    val statistics = new WorkflowUsageStatistics
    statistics.workflowsUsedLast180Days = calculateWorkflowUsageLast180Days()
    val workflowCountByStatus = repository.countReleasesByStatus(ReleaseKind.WORKFLOW).map(entry => entry.status -> entry.count).toMap
    statistics.startedWorkflowsTotal = workflowCountByStatus.getOrElse(ReleaseStatus.IN_PROGRESS.value(), 0)
    statistics.abortedWorkflowsTotal = workflowCountByStatus.getOrElse(ReleaseStatus.ABORTED.value(), 0)
    statistics.failedWorkflowsTotal = workflowCountByStatus.getOrElse(ReleaseStatus.FAILED.value(), 0)
    statistics.completedWorkflowsTotal = workflowCountByStatus.getOrElse(ReleaseStatus.COMPLETED.value(), 0)
    statistics.workflowsLast24Hours = repository.countReleasesInLastDays(1, ReleaseKind.WORKFLOW)
    statistics.workflowsLast30Days = repository.countReleasesInLastDays(30, ReleaseKind.WORKFLOW)
    statistics.workflowsCompletionRateMonthAgo = repository.calculateCompletionRateLastMonth(ReleaseKind.WORKFLOW)
    statistics.workflowTemplatesOfficialTotal = repository.countWorkflowTemplatesByType(true)
    statistics.workflowTemplatesCustomTotal = repository.countWorkflowTemplatesByType(false)
    statistics
  }

  private def calculateWorkflowUsageLast180Days(): util.List[String] = {
    val workflowsTemplates = repository.workflowTemplateDetails().map(w => (w.releaseId -> w)).toMap
    val startedWorkflows = repository.findWorkflowsStartedLastMonths(6).map(w => (w.releaseId -> w)).toMap
    val completedWorkflows = repository.findWorkflowsCompletedLastMonths(6).map(w => (w.releaseId -> w)).toMap

    val workflowTemplateUsageReportEntries = workflowsTemplates.map { case (releaseId, workflow) =>
      val workflowType = if (workflow.folderPath.startsWith("/FolderDefaultReleaseContent")) "O" else "C"
      val scmId = workflow.scmId match {
        case null => "NA"
        case _ => if (workflow.scmId.length >= 8) workflow.scmId.substring(0, 8) else workflow.scmId.substring(0, workflow.scmId.length - 1)
      }
      val numTimesStarted = startedWorkflows.get(releaseId).map(_.occurrences).getOrElse(0)
      val numUsersStartedWorkflow = startedWorkflows.get(releaseId).map(_.uniqueOwners).getOrElse(0)
      val numTimesCompleted = completedWorkflows.get(releaseId).map(_.occurrences).getOrElse(0)
      val numUsersCompletedWorkflow = completedWorkflows.get(releaseId).map(_.uniqueOwners).getOrElse(0)
      WorkflowTemplateUsageReportEntry(workflowType, workflow.title, scmId, numTimesStarted, numUsersStartedWorkflow, numTimesCompleted, numUsersCompletedWorkflow)
    }.toList

    workflowTemplateUsageReportEntries.sortBy(entry => (-entry.timesStarted, -entry.timesCompleted, entry.tile)).map(
      entry => s"${entry.workflowType}|${entry.tile}|${entry.scmId}|${entry.timesStarted}|${entry.usersStartCount}|" +
        s"${entry.timesCompleted}|${entry.usersCompletedCount}").asJava
  }
}

case class WorkflowTemplateUsageReportEntry(workflowType: String, tile: String, scmId: String, timesStarted: Int,
                                            usersStartCount: Int, timesCompleted: Int, usersCompletedCount: Int)

case class countByStatus(status: String, count: Int)

class ReleaseUsageStatistics {
  @BeanProperty
  var plannedReleasesTotal: Int = 0

  @BeanProperty
  var inProgressReleasesTotal: Int = 0

  @BeanProperty
  var pausedReleasesTotal: Int = 0

  @BeanProperty
  var failingReleasesTotal: Int = 0

  @BeanProperty
  var failedReleasesTotal: Int = 0

  @BeanProperty
  var completedReleasesTotal: Int = 0

  @BeanProperty
  var abortedReleasesTotal: Int = 0

  @BeanProperty
  var releasesLast24HoursTotal: Int = 0

  @BeanProperty
  var releasesLast30DaysTotal: Int = 0

  @BeanProperty
  var completionRateLastMonth: Float = 0

  @BeanProperty
  var averageReleasesPerDayLastYear: Float = 0

  @BeanProperty
  var averageReleasesPerMonthLastYear: Float = 0

  @BeanProperty
  var releasesWithDependencies: Int = 0

  @BeanProperty
  var dependenciesTotal: Int = 0

  @BeanProperty
  var averageDependenciesPerRelease: Float = 0

  @BeanProperty
  var maxDependenciesInRelease: Int = 0

  @BeanProperty
  var releaseTagsTotal: Int = 0

  @BeanProperty
  var averagePhasesPerRelease: Float = 0

  @BeanProperty
  var maxPhasesInRelease: Int = 0

  @BeanProperty
  var maxSecurableCiUidValue: Long = 0
}

class TaskUsageStatistics {
  @BeanProperty
  var plannedTasksTotal: Int = 0

  @BeanProperty
  var pendingTasksTotal: Int = 0

  @BeanProperty
  var inProgressTasksTotal: Int = 0

  @BeanProperty
  var completedTasksTotal: Int = 0

  @BeanProperty
  var skippedTasksTotal: Int = 0

  @BeanProperty
  var failedTasksTotal: Int = 0

  @BeanProperty
  var abortedTasksTotal: Int = 0

  @BeanProperty
  var templateTasksTotal: Int = 0

  @BeanProperty
  var averageTasksPerRelease: Float = 0

  @BeanProperty
  var maxTasksInRelease: Int = 0

  @BeanProperty
  var top25TaskTypes: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var automationPercentage: Float = 0

  @BeanProperty
  var taskTagsTotal: Int = 0

  @BeanProperty
  var taskBackupsTotal: Int = 0

  @BeanProperty
  var commentsTotal: Int = 0

  @BeanProperty
  var attachmentsTotal: Int = 0

  @BeanProperty
  var maxTasksCiUidValue: Long = 0
}

class UserUsageStatistics {
  @BeanProperty
  var usersTotal: Int = 0

  @BeanProperty
  var externalUsersTotal: Int = 0

  @BeanProperty
  var internalUsersTotal: Int = 0

  @BeanProperty
  var enabledUsersTotal: Int = 0

  @BeanProperty
  var activeUsersLastDay: Int = 0

  @BeanProperty
  var activeUsersLast30Days: Int = 0

  @BeanProperty
  var activeUsersLast90Days: Int = 0

  @BeanProperty
  var globalRolesTotal: Int = 0

  @BeanProperty
  var teamsTotal: Int = 0

  @BeanProperty
  var roleRolesTotal: Int = 0

  @BeanProperty
  var rolePrincipalsTotal: Int = 0

  @BeanProperty
  var rolePermissionsTotal: Int = 0
}

class TemplateUsageStatistics {
  @BeanProperty
  var templatesTotal: Int = 0

  @BeanProperty
  var templateRevisionsTotal: Int = 0
}

class TriggerUsageStatistics {
  @BeanProperty
  var triggersTotal: Int = 0

  @BeanProperty
  var activeTriggersTotal: Int = 0
}

class VariableUsageStatistics {
  @BeanProperty
  var globalVariablesTotal: Int = 0

  @BeanProperty
  var folderVariablesTotal: Int = 0

  @BeanProperty
  var averageVariablesPerFolder: Float = 0
}

class DashboardUsageStatistics {
  @BeanProperty
  var globalDashboardsTotal: Int = 0

  @BeanProperty
  var folderDashboardsTotal: Int = 0
}

class FolderUsageStatistics {
  @BeanProperty
  var foldersTotal: Int = 0

  @BeanProperty
  var maxDepthOfFolder: Int = 0
}

class ArchiveUsageStatistics {
  @BeanProperty
  var archivingEnabled: Boolean = false

  @BeanProperty
  var archivingAge: Int = 0

  @BeanProperty
  var archivedReleasesTotal: Int = 0

  @BeanProperty
  var prearchivedReleasesTotal: Int = 0

  @BeanProperty
  var oldestArchivedRelease: String = _

  @BeanProperty
  var archivedCompletedReleasesTotal: Int = 0

  @BeanProperty
  var archivedAbortedReleasesTotal: Int = 0

  @BeanProperty
  var archivedPhasesTotal: Int = 0

  @BeanProperty
  var archivedTasksTotal: Int = 0

  @BeanProperty
  var archivedAttachmentsTotal: Int = 0

  @BeanProperty
  var archivedTagsTotal: Int = 0

  @BeanProperty
  var archivedRolesTotal: Int = 0

  @BeanProperty
  var archivedMembersTotal: Int = 0

  @BeanProperty
  var workflowsArchivedCompleted: Int = 0

  @BeanProperty
  var workflowsArchivedAborted: Int = 0

  @BeanProperty
  var workflowsArchivedTotal: Int = 0

  @BeanProperty
  var workflowsPrearchivedTotal: Int = 0

  @BeanProperty
  var maxArchivedTasksUidValue: Long = 0
}

class DeliveryUsageStatistics {
  @BeanProperty
  var patternsTotal: Int = 0

  @BeanProperty
  var deliveriesTotal: Int = 0
}

class ReleaseGroupUsageStatistics {
  @BeanProperty
  var groupsTotal: Int = 0
}

class ReportUsageStatistics {
  @BeanProperty
  var activityLogsTotal: Int = 0

  @BeanProperty
  var attributesTotal: Int = 0

  @BeanProperty
  var deploymentsTotal: Int = 0

  @BeanProperty
  var deploymentHistoriesTotal: Int = 0

  @BeanProperty
  var deploymentRecordsTotal: Int = 0

  @BeanProperty
  var itsmRecordsTotal: Int = 0

  @BeanProperty
  var complianceRecordsTotal: Int = 0

  @BeanProperty
  var buildRecordsTotal: Int = 0

  @BeanProperty
  var planRecordsTotal: Int = 0

  @BeanProperty
  var archivedDeploymentRecordsTotal: Int = 0

  @BeanProperty
  var archivedItsmRecordsTotal: Int = 0

  @BeanProperty
  var archivedComplianceRecordsTotal: Int = 0

  @BeanProperty
  var archivedBuildRecordsTotal: Int = 0

  @BeanProperty
  var archivedPlanRecordsTotal: Int = 0

  @BeanProperty
  var permissionSnapshotsTotal: Int = 0

  @BeanProperty
  var riskAssessmentsTotal: Int = 0

  @BeanProperty
  var maxActivityLogIdValue: Long = 0
}

class ConfigurationUsageStatistics {
  @BeanProperty
  var globalConnectionsTotal: Int = 0

  @BeanProperty
  var folderConnectionsTotal: Int = 0
}

class ApplicationUsageStatistics {
  @BeanProperty
  var applicationsTotal: Int = 0
}

class EnvironmentUsageStatistics {
  @BeanProperty
  var environmentsTotal: Int = 0
}

class CalendarUsageStatistics {
  @BeanProperty
  var calendarEntriesTotal: Int = 0
}

class WorkflowUsageStatistics {
  @BeanProperty
  var workflowsUsedLast180Days: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var startedWorkflowsTotal: Int = 0

  @BeanProperty
  var abortedWorkflowsTotal: Int = 0

  @BeanProperty
  var failedWorkflowsTotal: Int = 0

  @BeanProperty
  var completedWorkflowsTotal: Int = 0

  @BeanProperty
  var workflowsLast24Hours: Int = 0

  @BeanProperty
  var workflowsLast30Days: Int = 0

  @BeanProperty
  var workflowsCompletionRateMonthAgo: Float = 0

  @BeanProperty
  var workflowTemplatesOfficialTotal: Int = 0

  @BeanProperty
  var workflowTemplatesCustomTotal: Int = 0
}
