package com.xebialabs.xlrelease.service;

import com.xebialabs.deployit.security.SecretKeyHolder;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.variable.VariableHelper;

/**
 * This is XLR specific password encrypter.
 *
 * Previous versions of XLR returned decrypted properties.
 *
 * This implementation will NOT return decrypted properties
 * unless the value is variable placeholder (i.e. contains ${...}),
 * or empty value (i.e. missing password is not a password).
 *
 * The idea is to have all properties encrypted by default and to decrypt them only in very isolated places
 * (i.e. when executing scripts of custom script tasks, script tasks, triggers or tile scripts).
 * In those rare places scripts should have access only to the password values of the current release
 * and globally shared configuration objects that are referenced (like JIRA servers and such).
 */
public class NonDecryptingPasswordEncrypter extends PasswordEncrypter {

    public NonDecryptingPasswordEncrypter() {
        super(SecretKeyHolder.get());
    }

    @Override
    public String ensureDecrypted(final String text) {
        String decryptedValue = super.ensureDecrypted(text);
        if (XlrConfig.getInstance().repository_decryptPasswords()) {
            return decryptedValue;
        } else {
            if (null == decryptedValue || decryptedValue.isEmpty() || VariableHelper.containsOnlyVariable(decryptedValue)) {
                return decryptedValue;
            } else {
                return ensureEncrypted(text);
            }
        }
    }
}
