package com.xebialabs.deployit.plumbing

import com.xebialabs.deployit.ClosingExceptionMapper._
import com.xebialabs.deployit.core.rest.resteasy.PathInterceptor
import com.xebialabs.deployit.{Exceptions, Sanitizer, ServerConfiguration}
import com.xebialabs.xlrelease.actors.RemoteException
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component

import java.util.{Date, UUID}
import jakarta.servlet.http.HttpServletRequest
import jakarta.ws.rs.core.HttpHeaders.ACCEPT
import jakarta.ws.rs.core.{Context, MediaType, Response}
import jakarta.ws.rs.ext.{ExceptionMapper, Provider}
import scala.jdk.CollectionConverters._

@Provider
@Component
class RemoteExceptionMapper(serverConfiguration: ServerConfiguration) extends ExceptionMapper[RemoteException] {
  @Context
  private var request: HttpServletRequest = _

  private val logger = LoggerFactory.getLogger(classOf[RemoteExceptionMapper])

  private val hideInternals = serverConfiguration.isHideInternals

  def getContent(mediaType: MediaType, exception: RemoteException) = {
    val rawContent = Exceptions.getAllMessages(exception)
    mediaType match {
      case mediaType if mediaType.isCompatible(MediaType.APPLICATION_JSON_TYPE) =>
        s"""{ "content": "${rawContent.replace("\"", "'")}" }"""
      case mediaType if mediaType.isCompatible(MediaType.APPLICATION_XML_TYPE) =>
        s"<![CDATA[${rawContent.replace("]]>", "___")}]]>"
      case _ =>
        rawContent
    }
  }

  override def toResponse(exception: RemoteException): Response = {
    handleLogging(exception)
    val responseStatusCode = exception.getResponseStatusCode
    val mediaType: MediaType = determineAcceptableMediaType()
    val content = getContent(mediaType, exception)

    if (hideInternals) {
      val uuid = UUID.randomUUID
      logger.error("[UUID={}] Exception: {}", uuid, exception.getClass.getName)
      logger.error("[UUID={}] Error messages: {}", uuid, content)
      Response.status(Response.Status.BAD_REQUEST)
        .`type`(MediaType.TEXT_PLAIN)
        .entity("An internal error has occurred, please notify your system administrator with the following code: " + uuid)
        .build
    } else {
      val builder = Response.status(responseStatusCode).`type`(mediaType)
      builder.header(X_EXCEPTION_TYPE, exception.getClass.getName)
      builder.header("X-Remote-Exception", "true")
      builder.header(X_PATH, PathInterceptor.PATH.get)
      builder.header("Date", new Date)

      builder.entity(Sanitizer.sanitize(content))
      builder.build
    }
  }

  private def handleLogging(exception: RemoteException): Unit = {
    logger.info(exception.getMsg)
    val logMsg = "Intercepting RemoteException"
    if (exception.getSuppressStackTrace()) {
      logger.trace(logMsg, exception)
    } else {
      logger.info(logMsg, exception)
    }
  }

  private def determineAcceptableMediaType(): MediaType = {
    val mediaTypes: List[MediaType] = if (request.getHeader("Accept-Type") != null) {
      List(MediaType.valueOf(request.getHeader("Accept-Type")))
    } else {
      if (request.getHeaders(ACCEPT) != null) {
        val accepted = request.getHeaders(ACCEPT)
        if (accepted != null) {
          accepted.asScala
            .flatMap(h => h.split(","))
            .map(e => MediaType.valueOf(e.trim))
            .toList
        } else {
          List()
        }
      } else {
        List(MediaType.APPLICATION_XML_TYPE)
      }
    }

    mediaTypes.find(MediaType.APPLICATION_JSON_TYPE.isCompatible)
      .getOrElse(mediaTypes.find(MediaType.APPLICATION_XML_TYPE.isCompatible)
        .getOrElse(MediaType.TEXT_PLAIN_TYPE)
      )
  }

}
