package com.xebialabs.xlrelease.support.report

import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.{Data, Seed}
import com.xebialabs.xlplatform.cluster.{ClusterMode, XlCluster}
import com.xebialabs.xlplatform.support.report.ReportDataProvider
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.apache.pekko.cluster.Cluster
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.beans.BeanProperty
import scala.concurrent.{Await, ExecutionContextExecutor}
import scala.jdk.CollectionConverters._

@Component
class ClusterDataProvider @Autowired()(val xlrConfig: XlrConfig,
                                       val actorSystemHolder: ActorSystemHolder)
  extends Logging with ReportDataProvider {

  val name: String = "cluster"

  override def collectStatistics: Map[String, Any] = {
    val statistics = new ClusterUsageStatistics

    val mode = xlrConfig.cluster.mode
    statistics.setClusterMode(mode.toString)

    if (mode == ClusterMode.Full) {
      statistics.setClustered(true)
      val actorSystem = actorSystemHolder.getInstance()
      val cluster = Cluster(actorSystem)
      implicit val context: ExecutionContextExecutor = actorSystem.dispatcher
      val management = XlCluster.get(actorSystem).membershipManagement
      try {
        val seeds = Await.result(management.listActiveSeeds(cluster), xlrConfig.timeouts.releaseActorReceive)
        val dbNodes = seeds match {
          case Data(seeds: Seq[Seed]) => seeds.map(_.address.toString)
          case _ => Seq.empty
        }
        statistics.setClusterDbNodes(dbNodes.toList.asJava)
        statistics.setClusterNumberOfNodes(dbNodes.length)
      } catch {
        case e: concurrent.TimeoutException => logger.warn("Timeout encountered querying active seeds, unable to collect all cluster statistics.")
        case _: Throwable => logger.warn("Exception encountered querying active seeds, unable to collect all cluster statistics.")
      }
      statistics.setClusterActorSystem(actorSystem.name)
      statistics.setClusterPekkoNodes(cluster.state.members.map(member => s"${member.address.toString} ${member.status}").toList.asJava)
      statistics.setClusterLeader(cluster.state.leader.map(_.toString).getOrElse("No leader"))
    }

    Map[String, Any]("cluster" -> statistics)
  }

}

class ClusterUsageStatistics {
  @BeanProperty
  var clusterMode: String = _

  @BeanProperty
  var clustered: Boolean = false

  @BeanProperty
  var clusterNumberOfNodes: Int = 0

  @BeanProperty
  var clusterActorSystem: String = _

  @BeanProperty
  var clusterLeader: String = _

  @BeanProperty
  var clusterDbNodes: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var clusterPekkoNodes: util.List[String] = new util.ArrayList[String]()
}
