package com.xebialabs.xlrelease.scheduler.logs

import com.google.common.annotations.VisibleForTesting
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.status.TaskStatus._
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionEntryDebounceActor.{DebounceTaskExecutionEntry, DebouncedEntry, FlushTaskExecutionEntries, TaskExecutionKey}
import com.xebialabs.xlrelease.service.{BroadcastService, TaskService}
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import grizzled.slf4j.Logging
import org.apache.pekko.actor.Actor

import java.net.URI
import scala.collection.mutable
import scala.concurrent.duration.FiniteDuration


object TaskExecutionEntryDebounceActor {

  final val name = "task-execution-entry-debounce-actor"


  protected case class DebouncedEntry(entry: TaskExecutionEntry, collectedEntryUris: List[URI]) {
    def merge(other: DebouncedEntry): DebouncedEntry = {
      DebouncedEntry(
        entry.mergeMax(other.entry),
        collectedEntryUris ++ other.collectedEntryUris
      )
    }
  }

  case class TaskExecutionKey(taskId: String, executionId: String) {
  }

  sealed trait LogDebounceMessages

  case class DebounceTaskExecutionEntry(key: TaskExecutionKey, entry: TaskExecutionEntry, collectedEntryUris: List[URI]) extends LogDebounceMessages

  case class FlushTaskExecutionEntries() extends LogDebounceMessages
}

@SpringActor
class TaskExecutionEntryDebounceActor(taskExecutionRepository: TaskExecutionRepository,
                                      taskService: TaskService,
                                      broadcastService: BroadcastService,
                                      xlrConfig: XlrConfig,
                                     ) extends Actor with Logging {

  import context._

  private val flushInterval: FiniteDuration = xlrConfig.features.taskExecutionEntryDebounceActor.flushInterval

  private lazy val taskExecutions: mutable.Map[TaskExecutionKey, DebouncedEntry] = mutable.Map()

  override def preStart(): Unit = {
    super.preStart()
    scheduleFlush()
  }

  @VisibleForTesting
  protected def scheduleFlush(): Unit = {
    system.scheduler.scheduleAtFixedRate(flushInterval, flushInterval, self, FlushTaskExecutionEntries())
  }

  override def postStop(): Unit = {
    flushTaskExecutions()
    super.postStop()
  }

  override def receive: Receive = {
    case DebounceTaskExecutionEntry(key, entry, collectedEntryUris) =>
      val debounced = DebouncedEntry(entry, collectedEntryUris)
      taskExecutions.updateWith(key) {
        case Some(existing) => Some(existing.merge(debounced))
        case None => Some(debounced)
      }.foreach { updated =>
        if (TaskExecutionLogService.DEFAULT_CHUNK_BUFFER_SIZE <= updated.entry.logSize) {
          flushTaskExecution(key, updated)
          taskExecutions.remove(key)
        }
      }
    case FlushTaskExecutionEntries() =>
      flushTaskExecutions()
  }

  private def flushTaskExecution(key: TaskExecutionKey, debounced: DebouncedEntry): Unit = {
    val taskStatus = taskService.getStatus(key.taskId)
    logger.debug(s"Adding log entry: $debounced, with task status: $taskStatus")
    //TODO: A new executionId is generated for abort script and failure handler script.
    // While it is not applicable for container based task, it would be nice to handle that here and avoid setting end-date for
    // ABORT_SCRIPT_IN_PROGRESS or FAILURE_HANDLER_IN_PROGRESS status.
    if (taskStatus.isOneOf(COMPLETED, SKIPPED, ABORTED, FAILED, FAILING, ABORT_SCRIPT_IN_PROGRESS, FAILURE_HANDLER_IN_PROGRESS)) {
      taskExecutionRepository.update(debounced.entry)
    } else {
      taskExecutionRepository.update(debounced.entry.copy(endDate = None))
    }
    broadcastService.broadcast(TaskLogCreated(key.taskId, key.executionId, debounced.collectedEntryUris), true)
  }

  private def flushTaskExecutions(): Unit = {
    taskExecutions.foreach(entry => flushTaskExecution(entry._1, entry._2))
    taskExecutions.clear()
  }
}
