package com.xebialabs.xlrelease.actors

import com.xebialabs.xlplatform.cluster.NodeState.isActive
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.service.BackgroundJobService
import com.xebialabs.xlrelease.utils.QuartzUtils.QuartzCron
import org.apache.pekko.actor._
import org.apache.pekko.extension.quartz.QuartzSchedulerExtension

import scala.concurrent.Future
import scala.util.{Failure, Success}

object BackgroundJobActor {
  case object Tick
  case object JobDone
}

abstract class BaseBackgroundJobActor(actorName: String,
                                      service: BackgroundJobService,
                                      xlrConfig: XlrConfig)
  extends Actor with ActorLogging {

  private val scheduler = QuartzSchedulerExtension(context.system)
  private implicit val executionContext = context.system.dispatcher

  override def preStart() = {
    val cron = service.getCronSchedule.asQuartzCron
    scheduler.createSchedule(
      name = actorName,
      cronExpression = cron
    )
    log.info(s"Scheduling '$actorName' using cron $cron")
    scheduler.schedule(actorName, self, BackgroundJobActor.Tick)
  }

  override def postStop(): Unit = {
    log.info(s"Stopping '$actorName'")
    scheduler.cancelJob(actorName)
    super.postStop()
  }

  override def receive: Actor.Receive = scheduleJob

  private def scheduleJob: Receive = {
    case BackgroundJobActor.Tick =>
      if (isActive) {
        log.debug(s"Starting '$actorName'")
        context.become(ignoreTicks)
        executeJob()
      }
  }

  private def ignoreTicks: Receive = {
    case BackgroundJobActor.Tick =>
      log.debug(s"Tick is ignored while '$actorName' is in progress")
    case BackgroundJobActor.JobDone =>
      context.become(scheduleJob)
      log.debug(s"The '$actorName' is ready");
  }

  private def executeJob(): Unit = {
    implicit val ec = xlrConfig.executors.auxiliaryExecutor.executionContext
    Future {
      service.executeJob()
    } onComplete {
      case Success(_) =>
        log.debug(s"'$actorName' done")
        self ! BackgroundJobActor.JobDone
      case Failure(ex) =>
        log.debug(s"Error while running '$actorName'")
        log.error(ex, ex.getMessage)
        self ! BackgroundJobActor.JobDone
    }
  }
}
