package com.xebialabs.xlrelease.views.converters;

import java.util.ArrayList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.xlrelease.api.v1.views.TeamMemberView;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Team;

import static com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType.PRINCIPAL;
import static com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType.ROLE;
import static com.xebialabs.xlrelease.builder.TeamBuilder.newTeam;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.DUMMY;
import static java.util.stream.Collectors.toList;

/**
 * Converter for {@link TeamView}
 */
@Component
public class TeamViewConverter {

    private final TeamMemberViewConverter teamMemberViewConverter;

    @Autowired
    public TeamViewConverter(TeamMemberViewConverter teamMemberViewConverter) {
        this.teamMemberViewConverter = teamMemberViewConverter;
    }

    public TeamView toView(Team team) {
        List<TeamMemberView> members = new ArrayList<>();
        members.addAll(team.getMembers().stream().map(t -> teamMemberViewConverter.toView(t, PRINCIPAL)).collect(toList()));
        members.addAll(team.getRoles().stream().map(role -> teamMemberViewConverter.toView(role, ROLE)).collect(toList()));

        TeamView teamView = new TeamView();
        teamView.setId(team.getId());
        teamView.setTeamName(team.getTeamName());
        teamView.setMembers(members);
        teamView.setPermissions(team.getPermissions().stream().filter(p -> !DUMMY.getPermissionName().equals(p)).collect(toList()));
        teamView.setSystemTeam(team.isSystemTeam());
        return teamView;
    }

    public Team toTeam(TeamView teamView) {
        // NB: systemTeam is ignored since it's a computed property
        List<String> principals = teamView.getMembers().stream().filter(member -> member.getType() == PRINCIPAL).map(TeamMemberView::getName).collect(toList());
        List<String> roles = teamView.getMembers().stream().filter(member -> member.getType() == ROLE).map(TeamMemberView::getName).collect(toList());
        return newTeam().withId(teamView.getId()).withTeamName(teamView.getTeamName()).withMembers(principals).withRoles(roles).withPermissions(teamView.getPermissions()).build();
    }

}
