package com.xebialabs.xlrelease.service

import com.github.benmanes.caffeine.cache.stats.CacheStats
import com.xebialabs.xlrelease.config.CacheManagementConstants._
import com.xebialabs.xlrelease.domain.distributed.events.DistributedClearCacheGroupEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import org.springframework.cache.caffeine.CaffeineCache
import org.springframework.cache.{Cache, CacheManager}
import org.springframework.context.ApplicationContext
import org.springframework.stereotype.Service

import java.util
import scala.jdk.CollectionConverters._

@Service
@EventListener
class CacheManagementService(broadcastService: BroadcastService, applicationContext: ApplicationContext) {

  private val cacheGroupToManagerMap: Map[String, Set[String]] = Map(
    "ci-references" -> Set(CI_REFERENCE_CACHE_MANAGER),
    "dashboards" -> Set(TILE_CACHE_MANAGER),
    "external-principal-data" -> Set(EXTERNAL_PRINCIPAL_DATA_CACHE_MANAGER),
    "notification-settings" -> Set(EMAIL_NOTIFICATION_CACHE_MANAGER),
    "releases" -> Set(RELEASE_CACHE_MANAGER),
    "reports" -> Set(REPORTS_CACHE_MANAGER),
    "runners" -> Set(RUNNER_CACHE_MANAGER_NAME),
    "security" -> Set(SECURITY_CACHE_MANAGER),
    "security-users" -> Set(SECURITY_USER_CACHE_MANAGER, SECURITY_USER_GROUP_CACHE_MANAGER),
  )

  lazy val cacheManagers: Map[String, CacheManager] = applicationContext.getBeansOfType(classOf[CacheManager]).asScala.toMap

  @AsyncSubscribe
  def onCacheGroupClear(event: DistributedClearCacheGroupEvent): Unit = {
    this.clearCacheGroup(event.cacheGroup, broadcast = false)
  }

  def listCacheGroups: util.List[String] = cacheGroupToManagerMap.keys.toList.sorted.asJava

  private def clearCaches(managersToClear: Set[String]): Unit = {
    managersToClear
      .map(managerName => cacheManagers.get(managerName))
      .filter(_.isDefined)
      .map(_.get)
      .foreach(manager => {
        manager.getCacheNames.asScala.foreach(cacheName => {
          val cache: Cache = manager.getCache(cacheName)
          if (null != cache) cache.clear()
        })
      })
  }

  def clearAllCacheGroups(broadcast: Boolean): Unit = {
    listCacheGroups.asScala.foreach(cacheGroup => this.clearCacheGroup(cacheGroup, broadcast))
  }

  def clearCacheGroup(cacheGroup: String, broadcast: Boolean): Unit = {
    val managersToClear = cacheGroupToManagerMap.getOrElse(cacheGroup, Set())
    if (managersToClear.isEmpty) {
      throw new LogFriendlyNotFoundException(s"Cache group '$cacheGroup' was not found")
    }
    this.clearCaches(managersToClear)
    if (broadcast) {
      broadcastService.broadcast(DistributedClearCacheGroupEvent(cacheGroup), publishEventOnSelf = false)
    }
  }

  def cacheStats(cacheGroup: String): util.Map[String, util.Map[String, String]] = {
    val managers = cacheGroupToManagerMap.getOrElse(cacheGroup, Set())

    managers
      .map(managerName => cacheManagers.get(managerName))
      .filter(_.isDefined)
      .map(_.get)
      .flatten(manager => {
        manager.getCacheNames.asScala.map(cacheName => {
          val cache: Cache = manager.getCache(cacheName)
          val caffeineCacheStats: CacheStats = cache.asInstanceOf[CaffeineCache].getNativeCache.stats()
          val stats: util.Map[String, String] = new util.HashMap[String, String]();
          stats.put("averageLoadPenalty", String.valueOf(caffeineCacheStats.averageLoadPenalty))
          stats.put("evictionCount", String.valueOf(caffeineCacheStats.evictionCount))
          stats.put("hitCount", String.valueOf(caffeineCacheStats.hitCount))
          stats.put("hitRate", String.valueOf(caffeineCacheStats.hitRate))
          stats.put("requestCount", String.valueOf(caffeineCacheStats.requestCount))
          (cacheName, stats)
        })
      }).toMap.asJava
  }
}
