package com.xebialabs.xlrelease.principaldata;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.security.core.GrantedAuthority;

import static com.xebialabs.xlrelease.config.CacheManagementConstants.EXTERNAL_PRINCIPAL_DATA_CACHE_MANAGER;
import static com.xebialabs.xlrelease.config.PrincipalDataProviderCacheConstants.EXTERNAL_AUTHORITIES_CACHE;
import static com.xebialabs.xlrelease.config.PrincipalDataProviderCacheConstants.EXTERNAL_GROUP_EMAILS_CACHE;
import static com.xebialabs.xlrelease.config.PrincipalDataProviderCacheConstants.EXTERNAL_USERS_CACHE;
import static com.xebialabs.xlrelease.principaldata.UserData.NOT_FOUND;


@CacheConfig(cacheManager = EXTERNAL_PRINCIPAL_DATA_CACHE_MANAGER)
public class PrincipalDataProviderImpl implements PrincipalDataProvider {

    private static final Logger logger = LoggerFactory.getLogger(PrincipalDataProviderImpl.class);

    private final List<UserDataProvider> userProviders;

    private final List<LdapGroupEmailProvider> groupEmailProviders;

    public PrincipalDataProviderImpl() {
        this.userProviders = new ArrayList<>();
        this.groupEmailProviders = new ArrayList<>();
    }

    public void addUserProvider(UserDataProvider provider) {
        this.userProviders.add(provider);
    }

    public void addUserProvider(List<UserDataProvider> providers) {
        this.userProviders.addAll(providers);
    }

    public void addGroupProvider(LdapGroupEmailProvider provider) {
        this.groupEmailProviders.add(provider);
    }

    public void addGroupProvider(List<LdapGroupEmailProvider> providers) {
        this.groupEmailProviders.addAll(providers);
    }

    @Override
    @Cacheable(cacheNames = {EXTERNAL_USERS_CACHE}, unless = "#result.found == false")
    public UserData getUserData(final String username) {
        return this.userProviders.stream()
                .filter(provider -> provider.getUserData(username) != NOT_FOUND)
                .map(provider -> provider.getUserData(username))
                .findFirst()
                .orElse(NOT_FOUND);
    }

    @Override
    @Cacheable(cacheNames = {EXTERNAL_GROUP_EMAILS_CACHE}, unless = "#result == null")
    public String getGroupEmail(final String groupName) {
        return this.groupEmailProviders.stream()
                .filter(provider -> provider.getGroupEmail(groupName) != null)
                .map(provider -> provider.getGroupEmail(groupName))
                .findFirst()
                .orElse(null);
    }

    @Override
    @Cacheable(cacheNames = {EXTERNAL_AUTHORITIES_CACHE}, unless = "#result.empty")
    public Collection<? extends GrantedAuthority> getAuthorities(final String username) {
        return this.userProviders.stream()
                .filter(provider -> !provider.getAuthorities(username).isEmpty())
                .map(provider -> provider.getAuthorities(username))
                .findFirst()
                .orElse(Collections.emptySet());
    }

    @CacheEvict(cacheNames = {EXTERNAL_USERS_CACHE, EXTERNAL_AUTHORITIES_CACHE})
    public void invalidate(String username) {
        logger.debug("Invalidating user data for user: {}", username);
    }
}

