package com.xebialabs.xlrelease.features

import com.xebialabs.xlrelease.config.CacheManagementConstants.{RELEASE_CACHE_MANAGER, SECURITY_CACHE_MANAGER}
import com.xebialabs.xlrelease.domain.distributed.events.{DistributedConfigurationCreatedEvent, DistributedConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.events.AsyncSubscribe
import com.xebialabs.xlrelease.features.distributed.BaseDistributedFeature
import com.xebialabs.xlrelease.features.settings.CacheManagementFeatureSettings
import com.xebialabs.xlrelease.support.cache.caffeine.spring.XlrCaffeineCacheManager

trait CacheManagementFeature extends ToggleFeature

object CacheManagementFeature extends BaseDistributedFeature[CacheManagementFeatureSettings] with CacheManagementFeature {
  private val DEFAULT_ENABLED_STATUS = true
  private val RELEASE_PREFIX = "release"
  private val SECURITY_PREFIX = "security"

  private lazy val releaseCacheManager: XlrCaffeineCacheManager = springBean[XlrCaffeineCacheManager](RELEASE_CACHE_MANAGER)
  private lazy val securityCacheManager: XlrCaffeineCacheManager = springBean[XlrCaffeineCacheManager](SECURITY_CACHE_MANAGER)

  override def isEnabled(cacheName: Option[String]): Boolean = cacheName match {
    case Some(v) => isCacheEnabled(v)
    case None => DEFAULT_ENABLED_STATUS
  }

  private def isCacheEnabled(cacheName: String): Boolean = cacheName match {
    case name if name.startsWith(RELEASE_PREFIX) => featureSettings.releaseCacheEnabled
    case name if name.startsWith(SECURITY_PREFIX) => featureSettings.securityCacheEnabled
    case _ => DEFAULT_ENABLED_STATUS
  }

  @AsyncSubscribe
  override def onFeatureCreate(event: DistributedConfigurationCreatedEvent): Unit = {
    super.onFeatureCreate(event)
    updateCacheStates()
  }

  @AsyncSubscribe
  override def onFeatureUpdate(event: DistributedConfigurationUpdatedEvent): Unit = {
    super.onFeatureUpdate(event)
    updateCacheStates()
  }

  private def updateCacheStates(): Unit = {
    securityCacheManager.updateCachesState(featureSettings.securityCacheEnabled)
    releaseCacheManager.updateCachesState(featureSettings.releaseCacheEnabled)
  }
}
