package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import org.springframework.stereotype.Component;
import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xltype.serialization.json.JsonWriter;

import static com.xebialabs.deployit.plumbing.PropertyDescriptorJsonWriter.writePropertyDescriptor;
import static java.nio.charset.StandardCharsets.UTF_8;

@Component
@Provider
@Produces({MediaType.APPLICATION_JSON})
public class DescriptorJsonWriter implements MessageBodyWriter<Descriptor> {

    @Override
    public boolean isWriteable(final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return true;
    }

    @Override
    public long getSize(final Descriptor descriptor, final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return -1L;
    }


    @Override
    public void writeTo(final Descriptor descriptor, final Class<?> type, final java.lang.reflect.Type genericType, final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders, final OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(descriptor).getBytes(UTF_8));
    }

    @VisibleForTesting
    String toJson(Descriptor descriptor) {
        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writeJson(descriptor, writer);
        return stringWriter.toString();
    }

    public void writeJson(final Descriptor descriptor, final JsonWriter writer) {
        writer.object();
        descriptorProperties(writer, descriptor);
        writePropertyDescriptors(writer, descriptor);
        writeSuperTypes(writer, descriptor);
        writer.endObject();
    }

    protected static void descriptorProperties(JsonWriter writer, Descriptor descriptor) {
        writer.key("type").value(descriptor.getType().toString());
        writer.key("label").value(descriptor.getLabel());
        if (descriptor.getDeployableType() != null) {
            writer.key("deployableType").value(descriptor.getDeployableType().toString());
        }
        if (descriptor.getContainerType() != null) {
            writer.key("containerType").value(descriptor.getContainerType().toString());
        }
        writer.key("virtual").value(descriptor.isVirtual());
        if (!descriptor.isVirtual() && descriptor.getRoot().getRootNodeName() != null) {
            writer.key("root").value(descriptor.getRoot().getRootNodeName());
        }
        writer.key("description").value(descriptor.getDescription());
    }

    private static void writePropertyDescriptors(JsonWriter writer, Descriptor descriptor) {
        writer.key("properties");
        writer.array();

        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            writePropertyDescriptor(writer, pd);
        }

        writer.endArray();
    }

    protected static void writeSuperTypes(JsonWriter writer, Descriptor descriptor) {
        writer.key("superTypes");
        writer.array();

        for (Type sc : descriptor.getSuperClasses()) {
            writer.value(sc.toString());
        }

        writer.endArray();
    }
}
