package com.xebialabs.xlrelease.views.converters;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.TaskDefinition;
import com.xebialabs.xlrelease.domain.tasks.TaskViewConverter;
import com.xebialabs.xlrelease.views.AttachmentView;
import com.xebialabs.xlrelease.views.TaskFullView;
import com.xebialabs.xlrelease.views.UserView;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.stream.Collectors;

import static com.xebialabs.xlrelease.views.Flag.toItemFlag;
import static com.xebialabs.xlrelease.views.converters.DurationViewConverter.fromDurationView;
import static java.util.stream.Collectors.toList;

public abstract class BaseTaskViewConverter<T extends Task> implements TaskViewConverter<T>, PlanItemConverter {

    private UserViewConverter userViewConverter;

    @Autowired
    public void setUserViewConverter(final UserViewConverter userViewConverter) {
        this.userViewConverter = userViewConverter;
    }

    @Override
    public TaskFullView toFullView(final T task, final List<Type> allowedTaskTypesForAuthenticatedUser) {

        final TaskFullView view = new TaskFullView();

        populatePlanItemView(view, task);

        view.setStatus(task.getStatus());
        view.setPrecondition(task.getPrecondition());
        view.setPreconditionEnabled(task.isPreconditionEnabled());
        view.setFailureHandler(task.getFailureHandler());
        view.setFailureHandlerEnabled(task.isFailureHandlerEnabled());
        view.setTaskFailureHandlerEnabled(task.isTaskFailureHandlerEnabled());
        view.setCheckAttributes(task.isCheckAttributes());
        view.setTaskRecoverOp(task.getTaskRecoverOp());
        view.setEditable(task.isUpdatable() && allowedTaskTypesForAuthenticatedUser.contains(task.getTaskType()));
        view.setFlag(toItemFlag(task));

        if (task.hasOwner()) {
            view.setOwner(userViewConverter.toUserView(task.getOwner()));
        }

        if (task.getPhase() != null) {
            view.setPhaseTitle(task.getPhase().getTitle());
        }

        Release release = task.getRelease();
        if (release != null) {
            view.setReleaseId(release.getId());
            view.setReleaseTitle(release.getTitle());
            String releaseOwner = release.getOwner();
            view.setReleaseOwner(userViewConverter.toUserView(releaseOwner));
        }
        view.setTeam(task.getTeam());
        view.setNumberOfComments(task.getComments().size());
        view.setAutomated(task.getProperty("automated"));
        view.setWaitForScheduledStartDate(task.isWaitForScheduledStartDate());
        view.setDelayDuringBlackout(task.isDelayDuringBlackout());
        if (task.getAttachments() != null) {
            view.setAttachments(task.getAttachments().stream().map(AttachmentView::new).collect(toList()));
        }
        view.setFlagsCount(task.getFlaggedCount());
        view.setDelaysCount(task.getDelayedCount());
        view.setFailuresCount(task.getFailuresCount());
        view.setTypeDisplayGroup(TaskDefinition.getDisplayGroup(task.getType()));
        view.setTypeDisplayName(TaskDefinition.getDisplayName(task.getType()));
        view.setVariableMapping(task.getVariableMapping());
        view.setTags(task.getTags());
        view.setConfigurationUri(task.getConfigurationUri());
        view.setPostponedDueToBlackout(task.isPostponedDueToBlackout());
        view.setPostponedUntilEnvironmentsAreReserved(task.isPostponedUntilEnvironmentsAreReserved());
        view.setOriginalScheduledStartDate(task.getOriginalScheduledStartDate());
        view.setLocked(task.isLocked());
        view.setDocumentationPage(task.getDocumentationPage());
        view.setFacets(task.getFacets());
        view.setWatchers(task.getWatchers().stream().map(username -> userViewConverter.toUserView(username))
                .collect(Collectors.toSet()));
        if (task.get$ciAttributes().getLastModifiedAt() != null) {
            view.setLastModifiedAt(task.get$ciAttributes().getLastModifiedAt().toDate());
        }
        view.setLastModifiedBy(task.get$ciAttributes().getLastModifiedBy());
        return view;
    }

    abstract protected Class<T> getTaskClass();

    abstract protected T fromView(final TaskFullView view);

    @Override
    public T toTask(final TaskFullView view) {
        T task = fromView(view);
        return fillCommonFields(view, task);
    }

    private T fillCommonFields(final TaskFullView view, final T task) {
        task.setId(view.getId());
        task.setTitle(view.getTitle());
        task.setDescription(view.getDescription());
        task.setScheduledStartDate(view.getScheduledStartDate());
        task.setDueDate(view.getDueDate());
        task.setStartDate(view.getStartDate());
        task.setEndDate(view.getEndDate());
        task.setPlannedDuration(fromDurationView(view.getPlannedDuration()));
        task.setStatus(view.getStatus());
        task.setPrecondition(view.getPrecondition());
        task.setFailureHandler(view.getFailureHandler());
        task.setTaskFailureHandlerEnabled(view.isTaskFailureHandlerEnabled());
        task.setTaskRecoverOp(view.getTaskRecoverOp());
        task.setFlagStatus(view.getFlag().getStatus());
        task.setFlagComment(view.getFlag().getComment());
        if (view.hasOwner()) {
            task.setOwner(view.getOwner().getUsername());
        }
        task.setTeam(view.getTeam());
        task.setWaitForScheduledStartDate(view.isWaitForScheduledStartDate());
        task.setDelayDuringBlackout(view.isDelayDuringBlackout());
        fillVariableMappings(view, task);
        task.setTags(view.getTags());
        task.setOriginalScheduledStartDate(view.getOriginalScheduledStartDate());
        task.setPostponedDueToBlackout(view.isPostponedDueToBlackout());
        task.setPostponedUntilEnvironmentsAreReserved(view.isPostponedUntilEnvironmentsAreReserved());
        task.setLocked(view.isLocked());
        task.setCheckAttributes(view.isCheckAttributes());
        task.setWatchers(view.getWatchers().stream().map(UserView::getUsername).collect(Collectors.toSet()));
        task.set$ciAttributes(new CiAttributes(null, null, view.getLastModifiedBy(),
                view.getLastModifiedAt() != null ? new DateTime(view.getLastModifiedAt()) : null, 0));
        return task;
    }

    protected void fillVariableMappings(final TaskFullView view, final T task) {
        if (!task.hasVariableMapping()) {
            task.setVariableMapping(view.getVariableMapping());
        }
    }

    @Override
    public Type getTaskType() {
        return Type.valueOf(getTaskClass());
    }

}
