package com.xebialabs.xlrelease.api.v1;

import java.util.*;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.v1.forms.CreateTeam;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.domain.events.TeamCreatedEvent;
import com.xebialabs.xlrelease.domain.events.TeamDeletedEvent;
import com.xebialabs.xlrelease.domain.events.TeamUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.CreateTeamOperation;
import com.xebialabs.xlrelease.service.TeamMembership;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.service.TeamUpdateOperation;
import com.xebialabs.xlrelease.views.converters.TeamViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.repository.Ids.isInRootFolder;
import static com.xebialabs.xlrelease.repository.Ids.isReleaseId;
import static scala.jdk.javaapi.CollectionConverters.asScala;

@Component
public class TeamFacade {

    private final PermissionChecker permissions;
    private final TeamService teamService;
    private final TeamViewConverter teamViewConverter;
    private final ReleaseActorService releaseActorService;
    private final XLReleaseEventBus eventBus;
    private static final List<String> SYSTEM_TEAMS = Arrays.asList(Team.FOLDER_OWNER_TEAMNAME, Team.RELEASE_ADMIN_TEAMNAME, Team.TEMPLATE_OWNER_TEAMNAME);

    @Autowired
    public TeamFacade(PermissionChecker permissions,
                      TeamService teamService,
                      TeamViewConverter teamViewConverter,
                      ReleaseActorService releaseActorService,
                      XLReleaseEventBus eventBus) {
        this.permissions = permissions;
        this.teamService = teamService;
        this.teamViewConverter = teamViewConverter;
        this.releaseActorService = releaseActorService;
        this.eventBus = eventBus;
    }

    public List<TeamView> getTeams(String containerId) {
        permissions.checkViewTeams(containerId);
        return teamService.getPublicTeamViews(containerId);
    }

    public List<TeamView> setTeams(String containerId, List<TeamView> teamsToUpdate) {
        checkNotNull(teamsToUpdate, "No teams submitted for update.");

        List<Team> submittedTeams = teamsToUpdate.stream().map(teamViewConverter::toTeam).toList();

        permissions.checkEditTeamsAgainstExisting(containerId, submittedTeams);
        checkPermissions(teamsToUpdate);

        if (isReleaseId(containerId)) {
            releaseActorService.updateTeams(containerId, submittedTeams);
        } else {
            if (isInRootFolder(containerId) || !teamsToUpdate.isEmpty()) {
                checkSystemTeamsArePresent(teamsToUpdate);
            }
            this.teamService.saveTeamsToPlatform(containerId, submittedTeams);
        }
        return teamService.getPublicTeamViews(containerId);
    }

    public void checkPermissions(final List<TeamView> teamsToUpdate) {
        Set<String> permissionsToCheck = teamsToUpdate.stream()
                .flatMap(team -> team.getPermissions().stream())
                .collect(Collectors.toUnmodifiableSet());
        checkForUnknownPermissions(permissionsToCheck);
    }

    private void checkForUnknownPermissions(Set<String> permissions) {
        Set<String> unknown = permissions.stream()
                .filter(p -> Permission.find(p) == null)
                .collect(Collectors.toUnmodifiableSet());
        checkArgument(unknown.isEmpty(), "Unknown permissions found: '%s'",
                String.join(", ", unknown));
    }

    private void checkSystemTeamsArePresent(final List<TeamView> teamsToUpdate) {
        List<String> missingSystemTeams = new ArrayList<>(SYSTEM_TEAMS);
        List<String> updateTeams = teamsToUpdate.stream().map(TeamView::getTeamName).toList();
        missingSystemTeams.removeAll(updateTeams);
        checkArgument(missingSystemTeams.isEmpty(), "Cannot update teams with the following system teams missing: %s", missingSystemTeams);
    }

    private boolean containerInheritsSecurity(String containerId) {
        String containerSecureId = teamService.securedCis().getSecuredCi(containerId).getSecurityUid();
        String effectiveSecureId = teamService.securedCis().getEffectiveSecuredCi(containerId).getSecurityUid();

        return !containerSecureId.equals(effectiveSecureId);
    }

    private void checkContainer(final String containerId) {
        if (containerInheritsSecurity(containerId)) {
            throw new Checks.IncorrectArgumentException(String.format("Security in container %s is inherited", containerId));
        }
    }

    public Team getTeam(final String containerId, final String teamId) {
        permissions.checkViewTeams(containerId);
        checkContainer(containerId);
        var maybeId = teamService.securityRepository().findTeamById(containerId, teamId);
        if (maybeId.isDefined()) {
            return maybeId.get();
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public Team findTeam(final String containerId, final String teamName) {
        permissions.checkViewTeams(containerId);
        checkContainer(containerId);
        var maybeId = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (maybeId.isDefined()) {
            return maybeId.get();
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public String createTeam(final String containerId, final CreateTeam createTeam) {
        var membership = new TeamMembership(
                asScala(createTeam.getPrincipals()).toSet(),
                asScala(createTeam.getRoles()).toSet(),
                asScala(createTeam.getPermissions()).toSet());

        TeamUpdateOperation op = new CreateTeamOperation(containerId, createTeam.getName(), membership);
        permissions.checkEditTeams(containerId, asScala(List.of(op)).toSeq());
        checkContainer(containerId);
        Set<String> createTeamPermissions = createTeam.getPermissions().stream().collect(Collectors.toUnmodifiableSet());
        checkForUnknownPermissions(createTeamPermissions);
        String teamId = teamService.securityRepository().createTeam(containerId, createTeam.getName(),
                asScala(createTeam.getRoles()).toSet(),
                asScala(createTeam.getPrincipals()).toSet(),
                asScala(createTeamPermissions).toSet());

        Team team = teamService.securityRepository().getTeam(teamId);
        eventBus.publish(new TeamCreatedEvent(containerId, team));
        return teamId;
    }

    public void addPrincipalsToTeamById(final String containerId, final String teamId, final List<String> principalNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().addPrincipalsToTeam(teamId, asScala(principalNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void addPrincipalsToTeamByName(final String containerId, final String teamName, final List<String> principalNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().addPrincipalsToTeam(original.get().getId(), asScala(principalNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public void removePrincipalsFromTeamById(final String containerId, final String teamId, final List<String> principalNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().removePrincipalsFromTeam(teamId, asScala(principalNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void removePrincipalsFromTeamByName(final String containerId, final String teamName, final List<String> principalNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().removePrincipalsFromTeam(original.get().getId(), asScala(principalNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public void addRoleNamesToTeamById(final String containerId, final String teamId, final List<String> roleNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().addRolesByNameToTeam(teamId, asScala(roleNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void addRoleNamesToTeamByName(final String containerId, final String teamName, final List<String> roleNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().addRolesByNameToTeam(original.get().getId(), asScala(roleNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public void removeRoleNamesFromTeamById(final String containerId, final String teamId, final List<String> roleNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().removeRolesByNameFromTeam(teamId, asScala(roleNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void removeRoleNamesFromTeamByName(final String containerId, final String teamName, final List<String> roleNames) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamMembers(containerId, original.get()),
                    () -> teamService.securityRepository().removeRolesByNameFromTeam(original.get().getId(), asScala(roleNames).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public void addPermissionsToTeamById(final String containerId, final String teamId, final List<String> permissionsToAdd) {
        checkContainer(containerId);
        Set<String> p = permissionsToAdd.stream().collect(Collectors.toUnmodifiableSet());
        checkForUnknownPermissions(p);
        Set<String> removed = Collections.emptySet();

        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamPermissions(containerId, asScala(p).toSeq(), asScala(removed).toSeq()),
                    () -> teamService.securityRepository().addPermissionsToTeam(teamId, asScala(p).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void addPermissionsToTeamByName(final String containerId, final String teamName, final List<String> permissionsToAdd) {
        checkContainer(containerId);
        Set<String> p = permissionsToAdd.stream().collect(Collectors.toUnmodifiableSet());
        checkForUnknownPermissions(p);
        Set<String> removed = Collections.emptySet();

        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamPermissions(containerId, asScala(p).toSeq(), asScala(removed).toSeq()),
                    () -> teamService.securityRepository().addPermissionsToTeam(original.get().getId(), asScala(p).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public void removePermissionsFromTeamById(final String containerId, final String teamId, final List<String> permissionsToRemove) {
        checkContainer(containerId);
        Set<String> added = Collections.emptySet();

        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamPermissions(containerId, asScala(added).toSeq(), asScala(permissionsToRemove).toSeq()),
                    () -> teamService.securityRepository().removePermissionsFromTeam(teamId, asScala(permissionsToRemove).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void removePermissionsFromTeamByName(final String containerId, final String teamName, final List<String> permissionsToRemove) {
        checkContainer(containerId);
        Set<String> added = Collections.emptySet();

        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            updateTeamAndFireEvents(
                    containerId,
                    original.get(),
                    () -> permissions.checkEditTeamPermissions(containerId, asScala(added).toSeq(), asScala(permissionsToRemove).toSeq()),
                    () -> teamService.securityRepository().removePermissionsFromTeam(original.get().getId(), asScala(permissionsToRemove).toSet())
            );
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    public void deleteTeamById(final String containerId, final String teamId) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamById(containerId, teamId);
        if (original.isDefined()) {
            permissions.checkDeleteTeam(containerId, original.get());
            teamService.securityRepository().deleteTeam(teamId);
            eventBus.publish(new TeamDeletedEvent(containerId, original.get()));
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamId));
        }
    }

    public void deleteTeamByName(final String containerId, final String teamName) {
        checkContainer(containerId);
        var original = teamService.securityRepository().findTeamByName(containerId, teamName);
        if (original.isDefined()) {
            permissions.checkDeleteTeam(containerId, original.get());
            teamService.securityRepository().deleteTeam(original.get().getId());
            eventBus.publish(new TeamDeletedEvent(containerId, original.get()));
        } else {
            throw new NotFoundException(String.format("Team %s not found", teamName));
        }
    }

    private void updateTeamAndFireEvents(String containerId, Team originalTeam, Runnable permissionCheck, Runnable action) {
        permissionCheck.run();
        action.run();
        Team updatedTeam = teamService.securityRepository().getTeam(originalTeam.getId());
        if (originalTeam != updatedTeam) {
            eventBus.publish(new TeamUpdatedEvent(containerId, originalTeam, updatedTeam));
        }
    }
}
