package com.xebialabs.xlrelease

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.booter.local.LocalBooter
import com.xebialabs.deployit.engine.spi.event.MaintenanceStartEvent
import com.xebialabs.deployit.{ServerConfigFile, ServerState}
import com.xebialabs.license.LicenseRegistrationServlet
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.startup.{PluginSynchronizer, SourceOfTruth, SynchronizationCondition}
import com.xebialabs.xlrelease.XLReleaseBootstrapper.PRODUCT_DIGITALAI
import com.xebialabs.xlrelease.XLReleaseServerLaunchOptions.parseCommandLine
import com.xebialabs.xlrelease.booter.XlrBooter
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.plugin.classloading.XlrPluginClassLoader
import com.xebialabs.xlrelease.script.Jsr223EngineFactory
import com.xebialabs.xlrelease.script.jython.JythonEngineInstance
import com.xebialabs.xlrelease.security.XLReleasePermissions
import com.xebialabs.xlrelease.spring.configuration.{XlrBooterInitializer, XlrProfiles, XlrWebApplicationInitializer}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_PLUGIN_MANAGER
import com.zaxxer.hikari.{HikariConfig, HikariDataSource}
import grizzled.slf4j.Logging
import org.bouncycastle.jce.provider.BouncyCastleProvider
import org.springframework.beans.factory.config.AutowireCapableBeanFactory
import org.springframework.boot.autoconfigure.EnableAutoConfiguration
import org.springframework.boot.autoconfigure.cache.CacheAutoConfiguration
import org.springframework.boot.autoconfigure.context.LifecycleAutoConfiguration
import org.springframework.boot.autoconfigure.dao.PersistenceExceptionTranslationAutoConfiguration
import org.springframework.boot.autoconfigure.data.jdbc.JdbcRepositoriesAutoConfiguration
import org.springframework.boot.autoconfigure.data.jpa.JpaRepositoriesAutoConfiguration
import org.springframework.boot.autoconfigure.data.web.SpringDataWebAutoConfiguration
import org.springframework.boot.autoconfigure.groovy.template.GroovyTemplateAutoConfiguration
import org.springframework.boot.autoconfigure.gson.GsonAutoConfiguration
import org.springframework.boot.autoconfigure.http.HttpMessageConvertersAutoConfiguration
import org.springframework.boot.autoconfigure.jackson.JacksonAutoConfiguration
import org.springframework.boot.autoconfigure.jdbc.{DataSourceAutoConfiguration, DataSourceTransactionManagerAutoConfiguration}
import org.springframework.boot.autoconfigure.jms.JmsAutoConfiguration
import org.springframework.boot.autoconfigure.jms.activemq.ActiveMQAutoConfiguration
import org.springframework.boot.autoconfigure.jms.artemis.ArtemisAutoConfiguration
import org.springframework.boot.autoconfigure.ldap.LdapAutoConfiguration
import org.springframework.boot.autoconfigure.mustache.MustacheAutoConfiguration
import org.springframework.boot.autoconfigure.orm.jpa.HibernateJpaAutoConfiguration
import org.springframework.boot.autoconfigure.quartz.QuartzAutoConfiguration
import org.springframework.boot.autoconfigure.security.oauth2.client.reactive.ReactiveOAuth2ClientAutoConfiguration
import org.springframework.boot.autoconfigure.security.oauth2.client.servlet.OAuth2ClientAutoConfiguration
import org.springframework.boot.autoconfigure.security.oauth2.resource.reactive.ReactiveOAuth2ResourceServerAutoConfiguration
import org.springframework.boot.autoconfigure.security.oauth2.resource.servlet.OAuth2ResourceServerAutoConfiguration
import org.springframework.boot.autoconfigure.security.servlet.{SecurityAutoConfiguration, SecurityFilterAutoConfiguration, UserDetailsServiceAutoConfiguration}
import org.springframework.boot.autoconfigure.session.SessionAutoConfiguration
import org.springframework.boot.autoconfigure.task.{TaskExecutionAutoConfiguration, TaskSchedulingAutoConfiguration}
import org.springframework.boot.autoconfigure.web.client.RestTemplateAutoConfiguration
import org.springframework.boot.autoconfigure.web.servlet.error.ErrorMvcAutoConfiguration
import org.springframework.boot.autoconfigure.web.servlet.{HttpEncodingAutoConfiguration, MultipartAutoConfiguration, WebMvcAutoConfiguration}
import org.springframework.boot.autoconfigure.websocket.servlet.WebSocketServletAutoConfiguration
import org.springframework.boot.web.servlet.ServletRegistrationBean
import org.springframework.boot.{Banner, CommandLineRunner, SpringApplication}
import org.springframework.context.annotation._
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.util.ResourceUtils

import java.security.Security
import java.util
import java.util.{Collections, Properties}
import jakarta.servlet.MultipartConfigElement

object XLReleaseBootstrapper extends Logging {
  val PRODUCT_DIGITALAI = "Digital.ai Release"
  val LOGBACK_CONFIG_FILE = "logback.configurationFile"
  val MAINTENANCE_MODE_LOG_MESSAGE =
    """The system is booting in maintenance mode. The following features are disabled:
      |- automated triggers
      |- archiving
      |- archive purging
      |- email notifications
      |- non-admin access
      |- task recovery""".stripMargin

  var releaseServer: ReleaseServer = _

  def boot(args: Array[String]) = {
    ReleaseServer.outputReleaseInfo()
    ReleaseServer.verifyAddOpens()

    val launchOptions: XLReleaseServerLaunchOptions = parseCommandLine(args)

    if (launchOptions == null) {
      System.exit(0)
    }

    // load configuration files 1st so we can use XlrConfig
    val serverConfFile = ResourceUtils.getFile(XlrConfig.bootConfig.confFile_location)
    val confFile = new ServerConfigFile(serverConfFile)

    Security.addProvider(new BouncyCastleProvider)
    XLReleasePermissions.init()

    releaseServer = new ReleaseServer(launchOptions, confFile)
    releaseServer.prepareConfigs()
    validateNoPluginsOnClasspath()
    val pluginClassloader = XlrPluginClassLoader()
    Thread.currentThread().setContextClassLoader(pluginClassloader)

    val xlrConfig = XlrConfig.getInstance
    if (xlrConfig.maintenanceModeEnabled) {
      ServerState.getInstance().maintenanceStart(new MaintenanceStartEvent)
      logger.warn(MAINTENANCE_MODE_LOG_MESSAGE)
    }

    setSecurityProperties(xlrConfig)

    syncPlugins(xlrConfig)

    val jythonInitialization = initJython(xlrConfig, pluginClassloader)

    LocalBooter.boot()
    XlrBooter().boot()

    jythonInitialization.join()
  }

  private def setSecurityProperties(xlrConfig: XlrConfig): Unit = {
    xlrConfig.jvm.networkaddressCacheTtl
      .foreach {ttl => Security.setProperty(xlrConfig.Jvm.NETWORKADDRESS_CACHE_TTL, ttl) }
  }

  def syncPlugins(xlrConfig: XlrConfig): Unit = {
    def createDataSource() = {
      val cfg = new HikariConfig()
      cfg.setDriverClassName(xlrConfig.xlrRepositoryDriver)
      cfg.setJdbcUrl(xlrConfig.xlrRepositoryJdbcUrl)
      cfg.setUsername(xlrConfig.xlrRepositoryUsername)
      cfg.setPassword(xlrConfig.xlrRepositoryPassword)
      new HikariDataSource(cfg)
    }

    def createPluginSynchronizer(jdbcTemplate: JdbcTemplate) = {
      val pluginSource = XLReleaseServerLaunchOptions.getInstance().getPluginSource
      val sourceOfTruth = if (pluginSource != null) SourceOfTruth.withName(pluginSource) else SourceOfTruth.DATABASE
      new PluginSynchronizer(new SqlPluginRepository(jdbcTemplate), "xl-release", sourceOfTruth)
    }

    // init config wrapper
    ConfigWrapper.initWith(XLProduct.XLRelease)

    val dataSource = createDataSource()
    val jdbcTemplate = new JdbcTemplate(dataSource)
    val shouldSync = Environment.isDevelopment || new SynchronizationCondition(new JdbcTemplate(dataSource), XL_RELEASE_PLUGIN_MANAGER).shouldSync()

    (Thread.currentThread.getContextClassLoader, shouldSync) match {
      case (pluginClassLoader: XlrPluginClassLoader, true) =>
        logger.info("Attempting plugin synchronization...")
        val pluginSynchronizer = createPluginSynchronizer(jdbcTemplate)
        pluginClassLoader.syncPlugins(pluginSynchronizer)
      case _ => logger.info("Skipping plugin synchronization")
    }

    dataSource.close()
  }

  def initJython(xlrConfig: XlrConfig, pluginClassloader: XlrPluginClassLoader): Thread = {
    val previousTccl = Thread.currentThread().getContextClassLoader()
    try {
      if (xlrConfig.security.scripting.sandbox.jython.isolatePlugins) {
        val jythonEngineClassloader = classOf[Jsr223EngineFactory].getClassLoader
        Thread.currentThread().setContextClassLoader(jythonEngineClassloader)
      } else {
        Thread.currentThread().setContextClassLoader(pluginClassloader)
      }
      val jythonInitialization: Runnable = () => JythonEngineInstance(xlrConfig.cache.config)
      val jythonInitializationThread = new Thread(jythonInitialization, "jython-initialization")
      jythonInitializationThread.start()
      jythonInitializationThread
    } finally {
      Thread.currentThread().setContextClassLoader(previousTccl)
    }
  }

  def main(args: Array[String]): Unit = {
    boot(args)
    System.setProperty("liquibase.hub.mode", "off")
    val app = new SpringApplication(classOf[XLReleaseBootstrapper])
    app.setAllowBeanDefinitionOverriding(true)
    app.setBannerMode(Banner.Mode.OFF)
    app.setDefaultProperties(defaultProperties)
    app.addInitializers(new XlrWebApplicationInitializer(), new XlrBooterInitializer())
    app.run(args: _*)
  }

  private def defaultProperties: Properties = {
    val properties = new Properties()
    properties.put("xl.server.session.storage.enabled", XlrConfig.getInstance.server_session_storage_enabled)
    properties.put("xl.release.config", XlrConfig.defaultConfigName)

    //set spring-boot logging property
    if (isNullOrEmpty(System.getProperty(LOGBACK_CONFIG_FILE))) {
      properties.put("logging.config", "conf/logback.xml")
    } else {
      properties.put("logging.config", System.getProperty(LOGBACK_CONFIG_FILE))
      System.setProperty(LOGBACK_CONFIG_FILE, "")
    }
    //rest-easy
    properties.put("server.servlet.context-parameters.resteasy.async.job.service.enabled", true)
    properties.put("server.servlet.context-parameters.resteasy.document.expand.entity.references", false)
    properties
  }

  private def validateNoPluginsOnClasspath(): Unit = {
    val classPath = System.getProperty("java.class.path", ".")
    val classPathSeparator = System.getProperty("path.separator")
    if (classPath.split(classPathSeparator).exists(entry => isPluginEntry(entry))) {
      logger.error("Plugin related directories are on the server classpath -- the system may not start properly. " +
        "Please remove the following entries from the conf/xlr-wrapper-*.conf files: hotfix/plugins, plugins/__local__, " +
        "plugins/xlr-official. From Release 10.2 on, the plugin classpath is handled internally by the Plugin Manager. " +
        "Consult the documentation for more details at https://docs.digital.ai/bundle/devops-release-version-v.@doc.version@/page/release/how-to/upgrade-7.5.x-to-current/")
    }
  }

  private def isPluginEntry(entry: String): Boolean = {
    entry.toLowerCase.startsWith("hotfix/*") || entry.toLowerCase.startsWith("hotfix/plugins") || entry.toLowerCase.startsWith("plugins")
  }
}

@Profile(Array("!" + XlrProfiles.INTEGRATION_TEST))
@Configuration
@EnableAutoConfiguration(
  exclude = Array(
    classOf[ActiveMQAutoConfiguration],
    classOf[CacheAutoConfiguration],
    classOf[ErrorMvcAutoConfiguration],
    classOf[GroovyTemplateAutoConfiguration],
    classOf[DataSourceAutoConfiguration],
    classOf[DataSourceTransactionManagerAutoConfiguration],
    classOf[GsonAutoConfiguration],
    classOf[HttpEncodingAutoConfiguration],
    classOf[HttpMessageConvertersAutoConfiguration],
    classOf[JacksonAutoConfiguration],
    classOf[JmsAutoConfiguration],
    classOf[LdapAutoConfiguration],
    classOf[LifecycleAutoConfiguration],
    classOf[MultipartAutoConfiguration],
    classOf[MustacheAutoConfiguration],
    classOf[OAuth2ClientAutoConfiguration],
    classOf[OAuth2ResourceServerAutoConfiguration],
    classOf[PersistenceExceptionTranslationAutoConfiguration],
    classOf[QuartzAutoConfiguration],
    classOf[ReactiveOAuth2ClientAutoConfiguration],
    classOf[ReactiveOAuth2ResourceServerAutoConfiguration],
    classOf[RestTemplateAutoConfiguration],
    classOf[SecurityAutoConfiguration],
    classOf[SecurityFilterAutoConfiguration],
    classOf[SessionAutoConfiguration],
    classOf[SpringDataWebAutoConfiguration],
    classOf[TaskExecutionAutoConfiguration],
    classOf[TaskSchedulingAutoConfiguration],
    classOf[UserDetailsServiceAutoConfiguration],
    classOf[WebSocketServletAutoConfiguration],
    classOf[WebMvcAutoConfiguration],
    classOf[ArtemisAutoConfiguration],
    classOf[JdbcRepositoriesAutoConfiguration],
    classOf[JpaRepositoriesAutoConfiguration],
    classOf[HibernateJpaAutoConfiguration]
  )
)
@ImportResource(
  Array(
    "classpath:spring/xlrelease-context.xml",
    "classpath*:hotfix-context.xml"
  )
)
class XLReleaseBootstrapper extends CommandLineRunner with Logging {

  @Bean
  def multipartConfigElement: MultipartConfigElement = {
    val maxSize = XlrConfig.getInstance.server.upload.maxSizeBytes
    val maxRequestSize = maxSize * 2 // 2 times the max size to account for the overhead of the request

    // TODO check what directory location should be used for file upload
    new MultipartConfigElement(System.getProperty("java.io.tmpdir"), maxSize, maxRequestSize, 0)
  }

  @Bean
  def licenseRegistrationServlet(beanFactory: AutowireCapableBeanFactory): ServletRegistrationBean[LicenseRegistrationServlet] = {
    val servletRegistrationBean = new ServletRegistrationBean[LicenseRegistrationServlet]
    val licenseRegistrationServlet = new LicenseRegistrationServlet

    val path = "static/0/"
    val initParams = new util.HashMap[String, String]
    initParams.put(LicenseRegistrationServlet.PRODUCT_NAME, PRODUCT_DIGITALAI)
    initParams.put(LicenseRegistrationServlet.PRODUCT_LOGO_URL, path + "styles/img/logo_digital_ai_license.svg")
    initParams.put(LicenseRegistrationServlet.FONT_LOCATION_EOT, path + "styles/fonts/open-sans/OpenSans-Regular-webfont.eot")
    initParams.put(LicenseRegistrationServlet.FONT_LOCATION_TTF, path + "styles/fonts/open-sans/OpenSans-Regular-webfont.ttf")
    initParams.put(LicenseRegistrationServlet.FAVICON, path + "styles/img/favicon.ico")
    initParams.put(LicenseRegistrationServlet.FAVICON_TYPE, "image/x-icon")
    initParams.put(LicenseRegistrationServlet.ACTIVATION_URL, "https://xebialabs.com/products/xl-release/trial/activation")

    beanFactory.autowireBean(licenseRegistrationServlet)
    servletRegistrationBean.setServlet(licenseRegistrationServlet)
    servletRegistrationBean.setInitParameters(initParams)
    servletRegistrationBean.setUrlMappings(Collections.singletonList("/productregistration/*"))
    servletRegistrationBean
  }

  override def run(args: String*): Unit = {}
}
