package com.xebialabs.xlrelease.principaldata;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider;
import org.springframework.stereotype.Service;

import com.xebialabs.xlrelease.config.XlrConfig;

@Service
public class PrincipalDataProviderFactoryBean implements FactoryBean<PrincipalDataProvider>, ApplicationContextAware, InitializingBean {
    private static final Logger logger = LoggerFactory.getLogger(PrincipalDataProviderFactoryBean.class);

    private ApplicationContext context;
    private XlrConfig xlrConfig;

    private PrincipalDataProvider provider;

    @Override
    public void afterPropertiesSet() throws Exception {
        PrincipalDataProviderImpl principalDataProvider = new PrincipalDataProviderImpl();

        DefaultUserDataProvider defaultProvider = new DefaultUserDataProvider();
        // 1. Local user data provider has highest priority
        principalDataProvider.addUserProvider(defaultProvider);

        // 2. LDAP user data provider
        List<UserDataProvider> ldapUserDataProviderList = tryInitUserDataProviderFromSpringLdapProvider();
        if (!ldapUserDataProviderList.isEmpty()) {
            principalDataProvider.addUserProvider(ldapUserDataProviderList);
        }
        // Try init LdapGroupEmailProvider
        List<LdapGroupEmailProvider> ldapGroupEmailProviderList = tryInitGroupEmailProviderFromLdapProvider();
        if (ldapGroupEmailProviderList.isEmpty()) {
            ldapGroupEmailProviderList = tryInitGroupEmailProviderFromLdapGroupEmailProvider();
        }
        if (!ldapGroupEmailProviderList.isEmpty()) {
            principalDataProvider.addGroupProvider(ldapGroupEmailProviderList);
        }

        this.provider = new PrincipalDataProviderCache(principalDataProvider, xlrConfig);
    }

    @Override
    public PrincipalDataProvider getObject() throws Exception {
        return provider;
    }

    @Override
    public Class<?> getObjectType() {
        return PrincipalDataProvider.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    private List<LdapGroupEmailProvider> tryInitGroupEmailProviderFromLdapGroupEmailProvider() {
        Map<String, LdapGroupEmailProvider> ldapGroupEmailProviders = context.getBeansOfType(LdapGroupEmailProvider.class);
        List<LdapGroupEmailProvider> ldapGroupEmailProviderList = new ArrayList<>();
        if (!ldapGroupEmailProviders.isEmpty()) {
            try {
                logger.info("Found {} in spring context", LdapGroupEmailProvider.class.getSimpleName());
                ldapGroupEmailProviders.forEach((name, ldapGroupEmailProvider) -> ldapGroupEmailProviderList.add(ldapGroupEmailProvider));
            } catch (Exception e) {
                logger.warn("Error initializing from {}", LdapGroupEmailProvider.class.getSimpleName(), e);
            }
        }
        return ldapGroupEmailProviderList;
    }

    private List<LdapGroupEmailProvider> tryInitGroupEmailProviderFromLdapProvider() {
        Map<String, com.xebialabs.xlrelease.security.authentication.LdapAuthenticationProvider> ldapProviders =
                context.getBeansOfType(com.xebialabs.xlrelease.security.authentication.LdapAuthenticationProvider.class);
        List<LdapGroupEmailProvider> ldapGroupEmailProviderList = new ArrayList<>();
        if (!ldapProviders.isEmpty()) {
            try {
                logger.info("Found {} in spring context, initializing from LDAP",
                        com.xebialabs.xlrelease.security.authentication.LdapAuthenticationProvider.class.getSimpleName());
                ldapProviders.forEach((name, ldapProvider) -> {
                    if (ldapProvider.getGroupEmailProvider() != null) {
                        ldapGroupEmailProviderList.add(ldapProvider.getGroupEmailProvider());
                    }
                });
            } catch (Exception e) {
                logger.warn("Error initializing from {}", com.xebialabs.xlrelease.security.authentication.LdapAuthenticationProvider.class.getSimpleName(), e);
            }
        }
        return ldapGroupEmailProviderList;
    }

    /**
     * Tries to obtain an LDAP connection from a "standard" Spring Security configuration (as described in the XLDeploy
     * configuration manual).
     * <p>
     * Due to the Spring implementation, this requires some reflection.
     */
    private List<UserDataProvider> tryInitUserDataProviderFromSpringLdapProvider() {
        Map<String, com.xebialabs.xlrelease.security.authentication.LdapAuthenticationProvider> ldapProviders =
                context.getBeansOfType(com.xebialabs.xlrelease.security.authentication.LdapAuthenticationProvider.class);

        List<UserDataProvider> ldapUserDataProviderList = new ArrayList<>();
        if (!ldapProviders.isEmpty()) {
            try {
                logger.info("Found {} in spring context, initializing from LDAP", LdapAuthenticationProvider.class.getSimpleName());
                ldapProviders.forEach((name, ldapProvider) -> ldapUserDataProviderList.add(new LdapUserDataProvider(ldapProvider, ldapProvider.getExternalIdAttribute())));
            } catch (Exception e) {
                logger.warn("Error initializing from {}", LdapAuthenticationProvider.class.getSimpleName(), e);
            }
        }
        return ldapUserDataProviderList;
    }

    @Override
    public void setApplicationContext(ApplicationContext context) throws BeansException {
        this.context = context;
    }

    @Autowired
    public void setXlrConfig(final XlrConfig xlrConfig) {
        this.xlrConfig = xlrConfig;
    }
}
