package com.xebialabs.xlrelease.server.jetty

import ch.qos.logback.access.joran.JoranConfigurator
import ch.qos.logback.access.spi.{AccessContext, AccessEvent, IAccessEvent, ServerAdapter}
import ch.qos.logback.classic.ClassicConstants
import ch.qos.logback.core.util.OptionHelper
import org.eclipse.jetty.server.{Request, RequestLog, Response}
import org.eclipse.jetty.util.component.ContainerLifeCycle

import java.util
import jakarta.servlet.http.{HttpServletRequest, HttpServletResponse}

class XlrRequestLog(fileName: String) extends ContainerLifeCycle with RequestLog {
  private val oldConf = OptionHelper.getSystemProperty(ClassicConstants.CONFIG_FILE_PROPERTY)
  private val configurator = new JoranConfigurator
  private val lc = new AccessContext
  lc.setName("AccessLog")
  configurator.setContext(lc)
  try {
    OptionHelper.setSystemProperty(configurator, ClassicConstants.CONFIG_FILE_PROPERTY, fileName)
    configurator.doConfigure(fileName)
  } finally {
    if (oldConf != null) {
      OptionHelper.setSystemProperty(configurator, ClassicConstants.CONFIG_FILE_PROPERTY, oldConf)
    }
  }

  def log(request: Request, response: Response): Unit = {
    val accessEvent: IAccessEvent = new AccessEvent(lc,
      httpServletRequest(request),
      httpServletResponse(response),
      adapter(request, response)
    )
    lc.callAppenders(accessEvent)
  }

  def adapter(request: Request, response: Response): ServerAdapter = new ServerAdapter {
    override def getRequestTimestamp: Long = Request.getTimeStamp(request)

    override def getContentLength: Long = Response.getContentBytesWritten(response)

    override def getStatusCode: Int = response.getStatus

    override def buildResponseHeaderMap(): util.Map[String, String] = {
      import scala.jdk.CollectionConverters._
      val responseHeaderMap = new util.HashMap[String, String]()
      val httpFields = response.getHeaders.asScala
      for (field <- httpFields) {
        val key = field.getName
        val value = field.getValue
        responseHeaderMap.put(key, value)
      }
      responseHeaderMap
    }
  }

  def httpServletRequest(request: Request): HttpServletRequest = {
    new RequestWrapper(request)
  }

  def httpServletResponse(response: Response): HttpServletResponse = {
    new ResponseWrapper(response)
  }

}
