package com.xebialabs.xlrelease.utils

import java.lang.{Boolean => JBoolean}
import java.security.{AccessControlContext, AccessController, PrivilegedAction}
import java.util.concurrent.ThreadFactory
import java.util.concurrent.atomic.AtomicInteger

object PrefixedThreadFactory {
  def apply(prefix: String, daemonic: JBoolean = JBoolean.TRUE): PrefixedThreadFactory = new PrefixedThreadFactory(prefix, daemonic)
}

/**
 * Copy from PrivilegedThreadFactory
 * @param prefix
 * @param daemonic
 */
class PrefixedThreadFactory(prefix: String, daemonic: JBoolean) extends ThreadFactory {
  private val poolNumber: AtomicInteger = new AtomicInteger(1)
  private val threadNumber: AtomicInteger = new AtomicInteger(1)
  private val s: SecurityManager = System.getSecurityManager
  private val group: ThreadGroup = if (s != null) s.getThreadGroup else Thread.currentThread.getThreadGroup
  private val namePrefix: String = prefix + "-pool-" + poolNumber.getAndIncrement + "-thread-"

  if (s != null) { // Calls to getContextClassLoader from this class
    // never trigger a security check, but we check
    // whether our callers have this permission anyways.
    s.checkPermission(new RuntimePermission("getClassLoader"))
    // Fail fast
    s.checkPermission(new RuntimePermission("setContextClassLoader"))
  }

  private val acc: AccessControlContext = AccessController.getContext
  private val ccl: ClassLoader = Thread.currentThread.getContextClassLoader

  override def newThread(r: Runnable): Thread = {
    val t = new Thread(group, r, namePrefix + threadNumber.getAndIncrement(), 0) {
      override def run(): Unit = {

        AccessController.doPrivileged(new PrivilegedAction[Void]() {
          override def run: Void = {
            Thread.currentThread.setContextClassLoader(ccl)
            r.run()
            null
          }
        }, acc)
      }
    }
    t.setDaemon(daemonic)
    t
  }
}
