package com.xebialabs.xlrelease.api.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.Part;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.domain.TemplateLogo;
import com.xebialabs.xlrelease.export.TemplateImportContext;
import com.xebialabs.xlrelease.export.TemplateImporter;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.UploadService;
import com.xebialabs.xlrelease.views.AttachmentView;
import com.xebialabs.xlrelease.views.ImportResult;
import com.xebialabs.xlrelease.views.TemplateLogoView;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.xlrelease.repository.Ids.isNullId;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;

/**
 * <p>
 * Handle file uploads used in various features of Digital.ai Release.
 * </p>
 * <p>
 * Upload URIs are not secured, because with IE the upload component falls back to an iframe, where we cannot pass authentication headers.
 * In order to limit the risk of exposing unsecured URIs, we make them temporary: before performing an upload, the client requests a
 * one-time token, that it must append to the URI it submits to.
 * </p>
 * Results must be TEXT_PLAIN because of IE9 iframe download handling, ZIP type makes IE believes it's a download of zip file.
 */
@Path("/upload")
@Consumes({MediaType.MULTIPART_FORM_DATA})
@Produces({MediaType.TEXT_HTML + "; charset=utf-8"})
@Controller
public class UploadResource {
    private static final Logger logger = LoggerFactory.getLogger(UploadResource.class);

    private static final String JSON_SUFFIX = ".json";

    private PermissionChecker permissionChecker;

    private TemplateImporter templateImporter;

    private UploadService uploadService;

    @Autowired
    public UploadResource(PermissionChecker permissionChecker, TemplateImporter templateImporter, UploadService uploadService) {
        this.permissionChecker = permissionChecker;
        this.templateImporter = templateImporter;
        this.uploadService = uploadService;
    }

    @SuppressWarnings("unused")
    public UploadResource() {
    }

    @POST
    @Timed
    @Path("templates/zip")
    public List<ImportResult> importTemplates(@Context HttpServletRequest request, @QueryParam("folderId") String folderId) throws IOException {
        if (isNullId(folderId)) {
            permissionChecker.check(XLReleasePermissions.CREATE_TEMPLATE);
        } else {
            permissionChecker.check(XLReleasePermissions.EDIT_TEMPLATE, folderId);
        }

        try {
            Optional<Part> templateInformation = request.getParts().stream().filter(this::filterFilePart).findFirst();
            if (templateInformation.isPresent()) {
                Part template = templateInformation.get();
                boolean isJson = template.getName().toLowerCase().endsWith(JSON_SUFFIX);
                boolean isDsl = template.getName().toLowerCase().endsWith(".zip");
                try (InputStream is = template.getInputStream()) {
                    return templateImporter.importTemplate(is, new TemplateImportContext(folderId, isJson,
                            false, isDsl, null));
                }
            } else {
                throw new IllegalArgumentException("Missing file");
            }
        } catch (ServletException e) {
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            if (rootCause instanceof IllegalStateException stateException) {
                throw stateException;
            }
            logger.error("Unable to upload file", e);
            throw new BadRequestException("Expected multipart content");
        } catch (IllegalStateException e) {
            logger.error("Unable to upload file", e);
            throw new BadRequestException(e);
        }
    }

    @POST
    @Timed
    @Path("attachment/{ciId}")
    public List<AttachmentView> addAttachments(@PathParam("ciId") @IdParam String ciId, @Context HttpServletRequest request) throws IOException {
        permissionChecker.checkEditAttachment(ciId);
        try {
            List<Part> attachmentParts = request.getParts().stream().filter(this::filterFilePart).toList();
            List<AttachmentView> returnedViews = new ArrayList<>();
            for (Part attachmentPart : attachmentParts) {
                String filename = attachmentPart.getSubmittedFileName();
                String contentType = attachmentPart.getContentType();
                InputStream inputStream = attachmentPart.getInputStream();
                returnedViews.add(new AttachmentView(this.uploadService.addAttachment(ciId, filename, contentType, inputStream)));
            }
            if (returnedViews.isEmpty()) {
                throw new BadRequestException("Expected file upload");
            }
            return returnedViews;
        } catch (ServletException e) {
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            if (rootCause instanceof IllegalStateException stateException) {
                throw stateException;
            }
            logger.error("Unable to upload file", e);
            throw new BadRequestException("Expected multipart content");
        } catch (IllegalStateException e) {
            logger.error("Unable to upload file", e);
            throw new BadRequestException(e);
        }
    }

    @POST
    @Timed
    @Path("logo/{templateId:.*Release[^/-]*}")
    public TemplateLogoView addLogo(@PathParam("templateId") @IdParam String templateId, @Context HttpServletRequest request) throws IOException {
        permissionChecker.check(EDIT_TEMPLATE, templateId);
        try {
            var logoList = request.getParts().stream().filter(this::filterFilePart).toList();
            int logoCount = logoList.size();
            if (logoCount > 1) {
                throw new BadRequestException("Can not upload more than one file");
            } else if (logoCount == 0) {
                throw new BadRequestException("No logo file provided");
            } else {
                Part logoPart = logoList.get(0);
                if (logoPart != null) {
                    var filename = logoPart.getSubmittedFileName();
                    var contentType = logoPart.getContentType();
                    var content = logoPart.getInputStream();
                    var size = logoPart.getSize();
                    var maxLogoSizeInBytes = TemplateLogo.getDefaultMaxLogoSize() * 1024 * 1024;
                    if (size > maxLogoSizeInBytes) {
                        throw new BadRequestException("Can not upload a logo over " + TemplateLogo.getDefaultMaxLogoSize() + " megabyte(s)");
                    }
                    var uploaded = this.uploadService.addLogo(templateId, filename, contentType, content);
                    return new TemplateLogoView(uploaded);
                } else {
                    throw new BadRequestException("Unable to find logo in a multipart request");
                }
            }
        } catch (ServletException e) {
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            if (rootCause instanceof IllegalStateException stateException) {
                throw stateException;
            }
            logger.error("Unable to upload file", e);
            throw new BadRequestException(e);
        } catch (IllegalStateException e) {
            logger.error("Unable to upload file", e);
            throw new BadRequestException(e);
        }
    }

    private Boolean filterFilePart(Part part) {
        return part.getSubmittedFileName() != null && !part.getSubmittedFileName().isEmpty();
    }

}
