package com.xebialabs.xlrelease.domain.tasks.userinput;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;
import org.springframework.stereotype.Component;

import com.xebialabs.xlrelease.domain.Changes;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.UserInputTask;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.domain.tasks.task.DefaultTaskUpdater;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.ReleaseVariableUpdateOperation;

import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_CONFIGURATION;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_USER_INPUT_TASK_VARIABLES_VALUES;
import static java.util.stream.Collectors.toList;

@Component
public class UserInputTaskUpdater extends DefaultTaskUpdater {

    @Override
    public Class<? extends Task> getTaskClass() {
        return UserInputTask.class;
    }

    @Override
    public Changes update(final Task original, final Task updated, final Set<TaskUpdateDirective> updateDirectives) {
        Changes changes = super.update(original, updated, updateDirectives);
        boolean taskUpdated = false;

        if (hasDirectiveToUpdateAllProperties(original, updateDirectives) ||
                updateDirectives.contains(UPDATE_TASK_CONFIGURATION)) {
            changes.update(original);
            updateVariablesConfiguration((UserInputTask) original, (UserInputTask) updated);
            taskUpdated = true;
        }

        if (updateDirectives.contains(UPDATE_USER_INPUT_TASK_VARIABLES_VALUES)) {
            Release release = original.getRelease();
            List<Variable> updatedVariables = ((UserInputTask) updated).getVariables();
            Map<String, Variable> variablesByKeys = release.getVariablesByKeys();
            for (Variable updatedVariable : updatedVariables) {
                Variable oldVariable = variablesByKeys.get(updatedVariable.getKey());
                if (oldVariable != null && !isPasswordMasked(updatedVariable)) {
                    release.replaceVariable(oldVariable, updatedVariable);
                    changes.addOperation(new ReleaseVariableUpdateOperation(oldVariable, updatedVariable));
                    taskUpdated = true;
                }
            }
        }

        if (taskUpdated) {
            taskConcurrencyService.updateLastModifiedDetails(original);
        }

        return changes;
    }

    private boolean isPasswordMasked(Variable updatedVariable) {
        return updatedVariable.isPassword() && PasswordVerificationUtils.isMaskedPassword(updatedVariable.getValueAsString());
    }

    private void updateVariablesConfiguration(final UserInputTask original, final UserInputTask updated) {
        List<Variable> releaseVariables = original.getRelease().getVariables();
        original.setVariables(updated.getVariables().stream()
                .filter(releaseVariables::contains)
                .collect(toList()));
    }

}
