package com.xebialabs.xlrelease.actors

import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.{Job, JobQueue}
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.pekko.actor.{Actor, ActorLogging, ActorRef, Cancellable}

import scala.concurrent.duration.DurationInt

@SpringActor
class JobConfirmationActor(jobQueue: JobQueue) extends Actor with ActorLogging {
  import context.dispatcher

  def receive: Receive = onMessage(Map.empty)

  private def onMessage(states: Map[JobId, JobConfirmationContext]): Receive = {
    case msg: MonitorJobConfirmation =>
      val cancellable = context.system.scheduler.scheduleOnce(5.seconds, self, JobConfirmationTimeout(msg.job.id))
      val newState = JobConfirmationContext(msg.job, sender(), cancellable)
      context.become(onMessage(states + (msg.job.id -> newState)))

    case msg: JobConfirmationTimeout =>
      states.get(msg.jobId) match {
        case Some(state) =>
          log.warning(s"Reserved job was not confirmed within 5 seconds, un reserving Job: ${state.job}")
          jobQueue.unReserve(state.job)
          context.become(onMessage(states - msg.jobId))
        case None =>
          log.debug(s"Received confirmation timeout msg for Job ${msg.jobId}, but couldn't find it in the states map. It might have already been confirmed.")
      }

    case msg: JobConfirmed =>
      states.get(msg.jobId) match {
        case Some(state) =>
          state.timeoutCancellable.cancel()
          state.replyTo ! msg
          context.become(onMessage(states - msg.jobId))
        case None =>
          log.warning(s"Received confirmation for Job ${msg.jobId}, but couldn't find it in the states map. It might have already been cancelled.")
      }

    case msg =>
      log.error(s"Can't process $msg")
  }
}

case class JobConfirmationContext(job: Job, replyTo: ActorRef, timeoutCancellable: Cancellable)
case class MonitorJobConfirmation(job: Job)
case class JobConfirmed(jobId: JobId)
case class JobConfirmationTimeout(jobId: JobId)

