package com.xebialabs.deployit.plumbing.authorization;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.authorization.AuthorizationDecision;
import org.springframework.security.authorization.AuthorizationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.client.authentication.OAuth2AuthenticationToken;

import com.xebialabs.deployit.security.authentication.PersonalAuthenticationToken;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.security.authentication.RunnerAuthenticationToken;
import com.xebialabs.xlrelease.service.UserLastActiveActorService;
import com.xebialabs.xlrelease.service.UserProfileService;

import static com.xebialabs.xlrelease.security.XLReleasePermissions.ADMIN_USERNAME;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;

public class LoginAuthorizationManager<T> implements AuthorizationManager<T> {

    private static final Logger logger = LoggerFactory.getLogger(LoginAuthorizationManager.class);

    private final UserProfileService userProfileService;
    private final UserLastActiveActorService userLastActiveActorService;

    public LoginAuthorizationManager(final UserProfileService userProfileService,
                                     final UserLastActiveActorService userLastActiveActorService) {
        this.userProfileService = userProfileService;
        this.userLastActiveActorService = userLastActiveActorService;
    }

    @Override
    public AuthorizationDecision check(final Supplier<Authentication> authentication, final T object) {
        boolean granted = isGranted(authentication.get());
        return new AuthorizationDecision(granted);
    }

    private boolean isGranted(Authentication authentication) {
        return authentication != null && authentication.isAuthenticated() && isAuthorized(authentication);
    }

    private boolean isAuthorized(Authentication authentication) {
        String username = authentication.getName();
        logger.debug("Checking if login allowed for [{}]", username);

        if (authentication instanceof RunnerAuthenticationToken) {
            return true;
        }

        if (isAdmin(username)) {
            userLastActiveActorService.updateLastActive(ADMIN_USERNAME);
            return true;
        }

        UserProfile userProfile = userProfileService.findByUsername(username);

        if (userProfile != null && userProfile.isLoginAllowed()) {
            logger.debug("User [{}] is authorized for login", username);
            onAccessGranted(authentication, userProfile);
            return true;
        }

        logger.error("User [{}] is not authorized for login", username);
        return false;
    }

    private void onAccessGranted(Authentication authentication, UserProfile userProfile) {
        // Copy of LoginPermissionVoter.onAccessGranted in 23.1 or older version

        userLastActiveActorService.updateLastActive(userProfile.getCanonicalId());

        // TODO: this does not belong here
        // TODO: https://digitalai.atlassian.net/browse/REL-1768
        Object details = authentication.getDetails();
        String realUserName = userProfile.getName();

        var oldAuthorities = authentication.getAuthorities();
        var authenticatedUserAuthority = new SimpleGrantedAuthority(XLReleasePermissions.AUTHENTICATED_USER);
        Collection<? extends GrantedAuthority> newAuthorities = oldAuthorities;
        if (!oldAuthorities.contains(authenticatedUserAuthority)) {
            Collection<GrantedAuthority> modifiableAuthorities = new HashSet<>();
            modifiableAuthorities.add(authenticatedUserAuthority);
            modifiableAuthorities.addAll(oldAuthorities);
            newAuthorities = Collections.unmodifiableCollection(modifiableAuthorities) ;
        }

        if (authentication instanceof final OAuth2AuthenticationToken oldToken) {
            var newAuthentication = new OAuth2AuthenticationToken(oldToken.getPrincipal(), newAuthorities, oldToken.getAuthorizedClientRegistrationId());
            newAuthentication.setDetails(oldToken.getDetails());
            SecurityContextHolder.getContext().setAuthentication(newAuthentication);
            return;
        } else if (authentication instanceof final PersonalAuthenticationToken oldToken) {
            var newAuthentication = new PersonalAuthenticationToken(oldToken.getPrincipal(),
                    (String) oldToken.getCredentials(),
                    oldToken.getExpiryDate(),
                    newAuthorities,
                    oldToken.getGlobalPermissions(),
                    oldToken.getConfigurationItemsPermissions(),
                    oldToken.getConfigurationItems());
            newAuthentication.setDetails(oldToken.getDetails());
            SecurityContextHolder.getContext().setAuthentication(newAuthentication);
            return;
        }

        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(
                realUserName,
                authentication.getCredentials(),
                newAuthorities);
        token.setDetails(details);

        // TODO: this is broken and breaks the spring security API's
        SecurityContextHolder.getContext().setAuthentication(token);
    }
}
