package com.xebialabs.xlrelease.service

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.status.ReleaseStatus._
import com.xebialabs.xlrelease.domain.variables.reference.{UsagePoint, UserInputTaskUsagePoint}
import com.xebialabs.xlrelease.domain.variables.{GlobalVariables, PasswordStringVariable, Variable}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils._
import com.xebialabs.xlrelease.variable.VariableHelper._
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper._
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils

import java.util.{List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
class VariableService @Autowired()(configurationRepository: ConfigurationRepository,
                                   releaseRepository: ReleaseRepository,
                                   variableRepository: ReleaseVariableRepository,
                                   archivingService: ArchivingService,
                                   ciIdService: CiIdService,
                                   taskBackup: TaskBackup,
                                   eventBus: XLReleaseEventBus) extends Logging {

  @Timed
  def findByIdIncludingArchived(variableId: String): Variable = {
    if (isGlobalVariableId(variableId)) {
      findGlobalVariableById(variableId)
    } else if (variableRepository.exists(variableId)) {
      findById(variableId)
    } else if (archivingService.exists(variableId)) {
      archivingService.getVariable(variableId)
    } else {
      throw new NotFoundException(s"Variable [$variableId] does not exist in the repository or archive")
    }
  }

  @Timed
  def findById(variableId: String): Variable = {
    if (isGlobalVariableId(variableId)) {
      findGlobalVariableById(variableId)
    } else {
      findReleaseVariableById(variableId)
    }
  }

  private def findReleaseVariableById(variableId: String): Variable = variableRepository.findById(variableId)

  private def findGlobalVariableById(variableId: String): Variable = configurationRepository.read[Variable](variableId)

  @Timed
  def findByKey(variableName: String, releaseId: String): Variable = variableName match {
    case name if isNotBlank(name) => variableRepository.findByKey(name, releaseId)
    case _ => null
  }

  @Timed
  def addVariable(release: Release, variable: Variable): Variable = {
    logger.debug(s"Adding new variable with key [${variable.getKey}] to release [${release.getId}]")
    // D-23493 if variable or nested CIs have IDs - those should be nullified
    variable.setId(null)
    val variableValueProvider = variable.getValueProvider
    if (variableValueProvider != null) {
      variableValueProvider.setId(null)
      variableValueProvider.setVariable(variable)
    }

    if (variable.isPassword) {
      variable.asInstanceOf[PasswordStringVariable].checkExternalPasswordVariable()
    }
    replacePasswordPropertiesInCiIfNeeded(None, variable)

    release.addVariable(variable)
    fixUpVariableIds(release.getId, release.getVariables, ciIdService)
    if (!release.getVariableById(variable.getId).isPresent) {
      release.addVariable(variable)
    }
    // Triggers
    syncTriggerVars(release)

    val created = variableRepository.create(variable, release)
    eventBus.publish(ReleaseVariableCreatedEvent(variable))
    created
  }

  @Timed
  def updateReleaseVariables(release: Release, variables: JList[Variable]): Release = {
    updateReleaseVariables(release, variables, validate = true)
  }

  @Timed
  def updateReleaseVariables(release: Release, variables: JList[Variable], validate: Boolean): Release = {
    variables.forEach(variable => checkArgument(!isNullOrEmpty(variable.getKey), "Variable must have a 'key' field"))
    val originalVariables = release.getVariables
    val originalVariablesList = originalVariables.asScala

    if (validate) {
      variables.asScala.foreach(v => {
        if (v.isPassword) {
          v.asInstanceOf[PasswordStringVariable].checkExternalPasswordVariable()
        }
        replacePasswordPropertiesInCiIfNeeded(originalVariablesList.find(_.getName == v.getName), v)
      })
    }

    // check for deleted variables that are used outside of the UserInputTask
    val newVariableKeys = variables.asScala.map(_.getKey)
    val deletedVariables = originalVariablesList.filterNot((variable: Variable) => newVariableKeys.contains(variable.getKey))
    val usedVariables = deletedVariables.flatMap { variable =>
      val variableKey = withVariableSyntax(variable.getKey)
      val isVariableUsedOutsideUserInputTask = release.collectVariableReferences().asScala.find(_.getKey.equals(variableKey)).exists { ref =>
        ref.getUsagePoints.asScala.exists(!_.isInstanceOf[UserInputTaskUsagePoint])
      }
      if (isVariableUsedOutsideUserInputTask) {
        Some(variable)
      } else {
        None
      }
    }
    val usedVariablesTxt = usedVariables.map(v => s"'${v.getKey}'").mkString(", ")
    if (StringUtils.hasText(usedVariablesTxt)) {
      throw new IncorrectArgumentException(s"The following variable(s) are still used: $usedVariablesTxt. Replace them before you try to delete them.")
    }

    release.setVariables(variables)
    fixUpVariableIds(release.getId, release.getVariables, ciIdService)
    // Triggers
    syncTriggerVars(release)
    variableRepository.update(originalVariablesList.toSeq, variables.asScala.toSeq, release)
    eventBus.publish(ReleaseVariablesUpdatedEvent(originalVariables, variables))
    release
  }

  @Timed
  def updateVariable(release: Release, updated: Variable): Variable = {
    def replaceVariableUsages(oldVariable: Variable, updatedVariable: Variable): Unit = {
      val replacement = new VariableOrValue
      replacement.setVariable(updatedVariable.getKey)
      val updatedRelease: Release = releaseRepository.findById(release.getId)
      replaceVariable(updatedRelease, oldVariable, replacement)
    }

    logger.debug(s"Updating variable [${updated.getId}]")
    updated.checkValidity()
    val current = release.getVariableById(updated.getId).get
    checkSameType(current, updated)
    if (current.getKey != updated.getKey) {
      if (release.getStatus != TEMPLATE && release.getStatus != PLANNED) {
        throw new IllegalStateException(s"Cannot rename variable ${updated.getId} of an already started release")
      }
      checkRenameValidity(release.getVariables.asScala.toSeq, current, updated)
    }

    val updatedVariable = updateVariableAndProvider(release.getId, current, updated, release)
    eventBus.publish(ReleaseVariableUpdatedEvent(current, updated))

    if (current.getKey != updated.getKey) {
      replaceVariableUsages(current, updatedVariable)
    }

    updatedVariable
  }

  @Timed
  def findGlobalVariablesOrEmpty(): GlobalVariables = findGlobalVariables()

  private def findGlobalVariables(): GlobalVariables = {
    val globalVariables = new GlobalVariables
    globalVariables.setVariables(configurationRepository.findAllByType[Variable](Type.valueOf(classOf[Variable])))
    globalVariables
  }

  @Timed
  def addGlobalVariable(v: Variable): Variable = {
    logger.debug(s"Adding new global variable with key [${v.getKey}]")
    if (v.isPassword) {
      v.asInstanceOf[PasswordStringVariable].checkExternalPasswordVariable()
    }
    replacePasswordPropertiesInCiIfNeeded(None, v)

    val globalVars = findGlobalVariables()

    val newVariable = globalVars.addVariable(v)
    fixUpVariableIds(globalVars.getId, globalVars.getVariables, ciIdService)

    configurationRepository.create(newVariable)
    eventBus.publish(GlobalVariableCreatedEvent(newVariable, null))
    newVariable
  }

  @Timed
  def updateGlobalVariable(updated: Variable): Variable = {
    logger.debug(s"Updating global variable [${updated.getId}]")
    updated.checkGlobalVariableValidity()

    val current = findGlobalVariableById(updated.getId)
    checkSameType(current, updated)

    val globalVars = findGlobalVariables()
    if (current.getKey != updated.getKey) {
      checkRenameValidity(globalVars.getVariables.asScala.toSeq, current, updated)
    }

    val variable = updateVariableAndProvider(globalVars.getId, current, updated, null)
    eventBus.publish(GlobalVariableUpdatedEvent(current, variable, null))
    variable
  }

  @Timed
  def deleteGlobalVariable(variableId: String): Unit = {
    logger.debug(s"Deleting global variable [$variableId]")
    val variable = findGlobalVariableById(variableId)
    configurationRepository.delete(variableId)
    eventBus.publish(GlobalVariableDeletedEvent(variable, null))
  }

  @Timed
  def deleteVariable(release: Release, variableId: String): Unit = {
    val variable = release.getVariableById(variableId)
      .orElseThrow(() => new NotFoundException(s"Repository entity [$variableId] not found"))

    val variableKey = withVariableSyntax(variable.getKey)
    val isVariableUsedOutsideUserInputTask = release.collectVariableReferences().asScala.find(_.getKey.equals(variableKey)).exists { ref =>
      ref.getUsagePoints.asScala.exists(!_.isInstanceOf[UserInputTaskUsagePoint])
    }
    if (isVariableUsedOutsideUserInputTask) {
      throw new IncorrectArgumentException(s"The following variable is still used: ${variable.getKey}. Replace it before you try to delete it.")
    }

    release.removeVariable(variableId)
    release.getAllUserInputTasks.forEach { task =>
      taskBackup.removeVariable(task, variableId)
    }
    // Triggers
    syncTriggerVars(release)

    variableRepository.delete(variableId, release)

    eventBus.publish(ReleaseVariableDeletedEvent(variable))
  }

  @Timed
  def replaceVariable(release: Release, variable: Variable, replacement: VariableOrValue): Unit = {
    if (replacement.getVariable == null && replacement.getValue == null) {
      replacement.setValue(variable.getEmptyValue)
    } else if (variable.isPassword) {
      replacement.setValue(replacePasswordIfNeeded(variable.getValue, replacement.getValue))
    }

    if (replacement.getVariable != null) {
      replacement.setVariable(formatVariableIfNeeded(replacement.getVariable))
    }

    val cis: mutable.Set[ConfigurationItem] = for {
      varRef <- release.collectVariableReferences.asScala
      if withoutVariableSyntax(varRef.getKey) == withoutVariableSyntax(variable.getKey)
      up <- varRef.getUsagePoints.asScala
      u <- updateUsagePoint(up, variable, replacement)
    } yield u

    scanAndBuildNewVariables(release, release, ciIdService)
    // Triggers
    syncTriggerVars(release)

    variableRepository.replace(release, cis.toSeq)
    eventBus.publish(ReleaseVariableReplacedEvent(variable, replacement))
  }

  private def checkRenameValidity(variables: Seq[Variable], current: Variable, updated: Variable): Unit = {
    if (variables.map(_.getKey).contains(updated.getKey)) {
      throw new IllegalArgumentException(s"Cannot rename variable ${updated.getId} from ${current.getKey}" +
        s" to ${updated.getKey} because ${updated.getKey} already exists")
    }
  }

  private def updateVariableAndProvider(parentId: String, current: Variable, updated: Variable, release: Release): Variable = {
    val currentValueType = current.getType.getDescriptor.getPropertyDescriptor("value")
    val updatedValueType = updated.getType.getDescriptor.getPropertyDescriptor("value")

    if (!currentValueType.getKind.equals(updatedValueType.getKind)) {
      throw new IllegalArgumentException(s"Cannot change type of variable from ${current.getType} to ${updated.getType}")
    }

    if (updated.isPassword) {
      updated.asInstanceOf[PasswordStringVariable].checkExternalPasswordVariable()
    }

    replacePasswordPropertiesInCiIfNeeded(Some(current), updated)

    if (updated.getValueProvider != null) {
      updated.getValueProvider.setId("")
      updated.getValueProvider.setVariable(updated)
    }

    fixUpVariableIds(parentId, List(updated).asJava, ciIdService)
    if (Ids.isReleaseId(parentId)) {
      release.replaceVariable(current, updated)
      // Triggers
      syncTriggerVars(release)

      variableRepository.update(current, updated, release)
    } else {
      configurationRepository.update(updated)
    }
    updated
  }

  private def updateUsagePoint(up: UsagePoint, variable: Variable, replacement: VariableOrValue): Set[ConfigurationItem] = {
    up.replaceVariable(variable, replacement).asScala.toSet
  }

  private def syncTriggerVars(release: Release): Unit = {
    val templateId = release.getId
    val templateVars = release.getVariables.asScala.withFilter(_.getShowOnReleaseStart).map(v => CiCloneHelper.cloneCi(v)).asJava
    if (release.getStatus == ReleaseStatus.TEMPLATE) {
      eventBus.publishAndFailOnError(TemplateVariablesChangedEvent(templateId, templateVars))
    }
  }
}
