package com.xebialabs.xlrelease.mbeans;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.cache.caffeine.CaffeineCache;
import org.springframework.jmx.export.annotation.ManagedOperation;
import org.springframework.jmx.export.annotation.ManagedResource;
import org.springframework.stereotype.Component;
import com.github.benmanes.caffeine.cache.stats.CacheStats;

import com.xebialabs.xlrelease.principaldata.PrincipalDataProvider;
import com.xebialabs.xlrelease.principaldata.PrincipalDataProviderCache;

import static java.util.stream.Collectors.toList;

@Component
@ManagedResource(objectName = "com.xebialabs.xlrelease.settings:name=CacheControl", description = "Cache control")
public class CacheControl {
    private PrincipalDataProvider principalDataProvider;
    private List<? extends CacheManager> cacheManagers = Collections.emptyList();


    @Autowired
    public CacheControl(PrincipalDataProvider principalDataProvider) {
        this.principalDataProvider = principalDataProvider;
    }

    @Autowired(required = false)
    public void setCacheManagers(final List<? extends CacheManager> cacheManagers) {
        this.cacheManagers = cacheManagers;
    }

    @ManagedOperation(description = "Clear principal data cache")
    public void clearUserDataCache() {
        if (this.principalDataProvider instanceof PrincipalDataProviderCache) {
            ((PrincipalDataProviderCache) this.principalDataProvider).clearCache();
        }
    }

    @ManagedOperation(description = "List caches")
    public List<String> listCaches() {
        return this.cacheManagers.stream().flatMap(manager -> manager.getCacheNames().stream()).collect(toList());
    }

    @ManagedOperation(description = "Clear named cache")
    public void clearCache(String cacheName) {
        this.cacheManagers.stream().forEach(manager -> {
            if (manager.getCacheNames().contains(cacheName)) {
                Cache cache = manager.getCache(cacheName);
                if (null != cache) {
                    cache.clear();
                }
            }
        });
    }

    @ManagedOperation(description = "Fetch named cache statistics")
    public List<HashMap<String, String>> cacheStats(String cacheName) {
        return this.cacheManagers.stream().map(manager -> {
            HashMap<String, String> stats = new HashMap<>();
            if (manager.getCacheNames().contains(cacheName)) {
                Cache cache = manager.getCache(cacheName);
                if (cache instanceof CaffeineCache) {
                    CaffeineCache caffeineCache = (CaffeineCache) cache;
                    final CacheStats caffeineCacheStats = (caffeineCache.getNativeCache().stats());
                    stats.put("averageLoadPenalty", String.valueOf(caffeineCacheStats.averageLoadPenalty()));
                    stats.put("evictionCount", String.valueOf(caffeineCacheStats.evictionCount()));
                    stats.put("hitCount", String.valueOf(caffeineCacheStats.hitCount()));
                    stats.put("hitRate", String.valueOf(caffeineCacheStats.hitRate()));
                    stats.put("requestCount", String.valueOf(caffeineCacheStats.requestCount()));
                }
            }
            return stats;
        }).collect(Collectors.toList());
    }
}
