package com.xebialabs.xlrelease.scheduler.configuration

import com.xebialabs.xlplatform.cluster.ClusterMode
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.scheduler.JobQueue
import com.xebialabs.xlrelease.scheduler.service.DatabaseJobProvider
import com.xebialabs.xlrelease.service.ServiceStarter
import grizzled.slf4j.Logging
import org.springframework.context.SmartLifecycle
import org.springframework.context.annotation.{Configuration, Profile}
import org.springframework.scheduling.annotation.{EnableScheduling, SchedulingConfigurer}
import org.springframework.scheduling.config.ScheduledTaskRegistrar

import java.time.Duration

object JobQueueConfiguration {
  final val LIFECYCLE_PHASE = ServiceStarter.LIFECYCLE_PHASE + 5
}

@Configuration
@EnableScheduling
@Profile(Array(ClusterMode.HOT_STANDBY, ClusterMode.FULL))
class JobQueueConfiguration(jobQueue: JobQueue,
                            databaseJobProvider: DatabaseJobProvider,
                            xlrConfig: XlrConfig)
  extends SchedulingConfigurer
    with SmartLifecycle
    with Logging {

  private lazy val jobQueuePopulateInterval: Long = xlrConfig.durations.jobQueuePopulateInterval.toMillis

  @volatile
  private var running: Boolean = false

  override def start(): Unit = {
    logger.debug("starting job queue populator")
    running = true
  }

  override def stop(): Unit = {
    logger.debug("stopping job queue populator")
    running = false
  }

  override def isRunning: Boolean = running

  override def getPhase: Int = JobQueueConfiguration.LIFECYCLE_PHASE

  private def populateJobQueue(): Unit = {
    if (running && jobQueue.isRunning() && jobQueue.size() == 0) {
      logger.debug("Going to populate jobQueue with new jobs")
      databaseJobProvider.queueJobsFromDatabase()
    }
  }

  override def configureTasks(taskRegistrar: ScheduledTaskRegistrar): Unit = {
    taskRegistrar.addFixedDelayTask(new Runnable() {
      override def run(): Unit = {
        populateJobQueue()
      }
    }, Duration.ofMillis(jobQueuePopulateInterval))
  }
}

