package com.xebialabs.xlrelease.views.converters;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.springframework.beans.factory.annotation.Autowired;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.domain.Configuration;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.repository.IdType;
import com.xebialabs.xlrelease.views.BaseConfigurationItemView;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.xlrelease.repository.IdType.CONFIGURATION;
import static com.xebialabs.xlrelease.utils.PasswordVerificationUtils.PASSWORD_MASK;

abstract class BaseConfigurationItemViewConverter {

    protected final ConfigurationRepository configurationRepository;

    @Autowired
    public BaseConfigurationItemViewConverter(ConfigurationRepository configurationRepository) {
        this.configurationRepository = configurationRepository;
    }

    void fillItem(ConfigurationItem item, BaseConfigurationItemView view) {
        getPropertyDescriptors(item.getType()).forEach(pd -> {
            Object propertyValue = view.getProperty(pd.getName());
            if (propertyValue != null) {
                if (pd.getKind() == PropertyKind.CI) {
                    if (propertyValue instanceof String) {
                        String internalId = (String) propertyValue;
                        if (!internalId.contains("Configuration/Custom")) {
                            internalId = IdType.CONFIGURATION.convertToInternalId((String) propertyValue);
                        }
                        propertyValue = configurationRepository.read(internalId);
                        Type referencedType = ((Configuration) propertyValue).getType();
                        checkArgument(referencedType.getDescriptor().isAssignableTo(pd.getReferencedType()),
                                "Type of CI must be " + pd.getReferencedType().getName() + " got " + referencedType.getName()
                        );
                    } else if (propertyValue instanceof Map) {
                        // _maybe_ received map is a nested object - so try to turn Map into a BaseConfigurationItemView
                        propertyValue = configurationItemFromMap((Map) propertyValue);
                    }
                } else if (pd.getKind() == PropertyKind.SET_OF_STRING) {
                    propertyValue = newHashSet((Collection<?>) propertyValue);
                } else if (pd.getKind() == PropertyKind.STRING) {
                    if (pd.isPassword()) {
                        propertyValue = PasswordEncrypter.getInstance().ensureDecrypted(propertyValue.toString());
                    }
                } else if (pd.getKind() == PropertyKind.SET_OF_CI) {
                    List<Object> listOfCiRepresentations = (List<Object>) propertyValue;
                    propertyValue = new HashSet<Collection<?>>();
                    // check what is inside and act based on that thing...
                    for (Object ciRepresentation : listOfCiRepresentations) {
                        if (ciRepresentation instanceof Map) {
                            ConfigurationItem ci = configurationItemFromMap((Map) ciRepresentation);
                            ((Collection<ConfigurationItem>) propertyValue).add(ci);
                        } else {
                            // TODO CI is ID - nothing for now - otherwise we have to load that ID
                        }
                    }
                }
            }
            if (!pd.isReadonly()) {
                item.setProperty(pd.getName(), propertyValue);
            }
        });

        item.setId(view.getId());
    }

    private ConfigurationItem configurationItemFromMap(Map map) {
        ConfigurationItem item = null;
        String id = (String) map.get("id");
        String typeName = (String) map.get("type");
        String title = (String) map.get("title");
        if (null != typeName) {
            Type type = Type.valueOf(typeName);
            final Descriptor typeDescriptor = type.getDescriptor();
            item = typeDescriptor.newInstance(id);

            BaseConfigurationItemView view = new BaseConfigurationItemView() {
                private String id;

                @Override
                public String getId() {
                    return this.id;
                }

                @Override
                public void setId(final String id) {
                    this.id = id;
                }
            };
            view.setId(id);
            view.setType(typeName);
            view.setTitle(title);
            view.setProperties(map);

            fillItem(item, view);
        }

        return item;
    }

    void fillView(BaseConfigurationItemView view, ConfigurationItem instance) {
        view.setId(instance.getId());
        view.setType(instance.getType().toString());
        view.setTitle(instance.getProperty("title"));

        Map<String, Object> properties = view.getProperties();
        getPropertyDescriptors(instance.getType()).forEach(pd -> {
            String propertyName = pd.getName();
            if (pd.getKind() == CI) {
                Configuration configuration = instance.getProperty(propertyName);
                properties.put(propertyName, configuration != null ? CONFIGURATION.convertToViewId(configuration.getId()) : null);
            } else {
                if (pd.isPassword() && instance.getProperty(propertyName) != null) {
                    properties.put(propertyName, PASSWORD_MASK());
                } else {
                    properties.put(propertyName, instance.getProperty(propertyName));
                }
            }
        });
    }

    <T extends ConfigurationItem> T newInstance(BaseConfigurationItemView view) {
        Descriptor descriptor = view.type().getDescriptor();
        return descriptor.newInstance(view.getId());
    }

    Stream<PropertyDescriptor> getPropertyDescriptors(final Type type) {
        return type.getDescriptor().getPropertyDescriptors().stream().filter(pd -> !pd.isHidden());
    }
}
