package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.event.{EventBusHolder, ShutdownEvent}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.XLReleaseServerLaunchOptions
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.{BaseConfiguration, Trigger}
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, TriggerRepository}
import com.xebialabs.xlrelease.service.DisableWebhook.{DisableWebhookTrigger, disableWebhookConsumer, disableWebhookSource}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.{BufferedReader, IOException, InputStreamReader}

@Component
class MissingTypesChecker @Autowired()(val configurationRepository: ConfigurationRepository,
                                       val triggerRepository: TriggerRepository,
                                       serverLaunchOptions: XLReleaseServerLaunchOptions,
                                       xlrConfig: XlrConfig) extends WebhookMissingTypeChecker with Logging {

  @Timed
  def cleanup(): Unit = {
    val (obsoleteFlags, missingTypes) = configurationRepository.getAllTypes.filter(typeName => {
      val theType = Type.valueOf(typeName)
      !theType.exists() || theType.getDescriptor.isVirtual
    }).partition(typeName => typeName.endsWith("FeatureSettings"))

    if (obsoleteFlags.nonEmpty) {
      logger.info(s"Removing obsolete feature flags [${obsoleteFlags.mkString(", ")}]...")
      configurationRepository.deleteByTypes(obsoleteFlags)
    }

    if (missingTypes.nonEmpty) {
      val NL = System.lineSeparator
      logger.info("Missing types found, need to clean them up, asking user to confirm.")
      val msg = s"$NL*** WARNING ***${NL}We detected that we need to remove missing types from your database$NL" +
        s"All configuration instances of types [${missingTypes.mkString(", ")}] and it's references will be removed$NL" +
        s"Please ensure you have 'INFO' level logging configured.${NL}Please enter 'yes' if you want to continue [no]: "
      print(msg)
      val response = if (serverLaunchOptions.isForceRemoveMissingTypes || xlrConfig.upgrader.forceRemoveMissingTypes) {
        "yes"
      } else {
        read
      }
      if (!"yes".equalsIgnoreCase(response)) {
        EventBusHolder.publish(new ShutdownEvent)
        throw FixMissingTypesRejectedException(
          s"""Digital.ai Release Startup failed because there are missing type definitions.
             |
             |To migrate unknown types, please rerun with the '-force-remove-missing-types'
             |Otherwise, please install the plugin for the missing types [${missingTypes.mkString(", ")}]""".stripMargin)
      } else {
        logger.info(s"Removing all configuration instances of the types [${missingTypes.mkString(", ")}] and it's references...")
        configurationRepository.deleteByTypes(missingTypes)
      }
      logger.info(s"User response was: $response")
    }
  }

  @Timed
  def disableMissingWebhookEventTypes(): Unit = {
    val processorConfigsWithMissingEventType = findEventProcessorsWithMissingEventType().toSeq
    val webhookConfigsWithMissingEventType = findWebhookEndpointsWithMissingEventType().toSeq
    val triggersWithMissingEventType = findEventBasedTriggersWithMissingEventType().toSeq

    val getMessage = (configs: Seq[BaseConfiguration], triggers: Seq[Trigger]) => {
      if (configs.nonEmpty && triggers.nonEmpty) {
        s"Following configurations: [${configs.map(_.getTitle).mkString(",")}] and " +
          s"triggers: [${triggers.map(_.getTitle).mkString(",")}]"
      } else if (configs.nonEmpty) {
        s"Following configuration: [${configs.map(_.getTitle).mkString(",")}]"
      } else if (triggers.nonEmpty) {
        s"Following triggers: [${triggers.map(_.getTitle).mkString(",")}]"
      } else {
        ""
      }
    }

    val missingConfigsAndTriggersMsg = getMessage(processorConfigsWithMissingEventType ++ webhookConfigsWithMissingEventType,
      triggersWithMissingEventType)

    if (missingConfigsAndTriggersMsg.nonEmpty) {
      logger.info("Missing event type detected")
      val msg =
        s"""We detected some missing event types.
           |$missingConfigsAndTriggersMsg have missing event type.
           |Please enter 'yes' if you want to disable found configurations/triggers with missing event type [no]:
           |""".stripMargin
      print(msg)
      if ("yes".equalsIgnoreCase(read)) {
        logger.info(s"Disabling ${missingConfigsAndTriggersMsg.toLowerCase()}")
        processorConfigsWithMissingEventType.foreach(disable(disableWebhookConsumer))
        webhookConfigsWithMissingEventType.foreach(disable(disableWebhookSource))
        triggersWithMissingEventType.foreach(disable(DisableWebhookTrigger))
      }
    }
  }

  protected def read: String = {
    val stdin = new BufferedReader(new InputStreamReader(System.in))
    try {
      Option(stdin.readLine).fold("")(_.trim)
    } catch {
      case e: IOException =>
        throw new RuntimeException(e)
    }
  }

}

case class FixMissingTypesRejectedException(private val msg: String) extends Exception(msg)
