package com.xebialabs.xlrelease.security.authentication

import com.xebialabs.deployit.booter.local.utils.Strings.isBlank
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plumbing.authentication.InternalUser
import com.xebialabs.deployit.security.authentication.{AuthenticationFailureException, PersonalAuthenticationToken}
import com.xebialabs.deployit.security.{PermissionEnforcer, UserService}
import com.xebialabs.xlrelease.features.PersonalAccessTokenFeature
import com.xebialabs.xlrelease.principaldata.PrincipalDataProvider
import com.xebialabs.xlrelease.security.UserGroupService
import com.xebialabs.xlrelease.service.{UserLastActiveActorService, UserTokenService}
import com.xebialabs.xlrelease.utils.DateVariableUtils.printDate
import com.xebialabs.xlrelease.utils.TokenGenerator
import grizzled.slf4j.Logging
import org.springframework.security.authentication.{AuthenticationProvider, BadCredentialsException, UsernamePasswordAuthenticationToken}
import org.springframework.security.core.authority.SimpleGrantedAuthority
import org.springframework.security.core.authority.mapping.{GrantedAuthoritiesMapper, NullAuthoritiesMapper}
import org.springframework.security.core.{Authentication, AuthenticationException, GrantedAuthority}
import org.springframework.stereotype.Component

import java.util
import java.util.Collections

/**
 * Authentication provider to authenticate internal users with username & password or personal access token
 */
@Component("xlAuthenticationProvider")
class ReleaseAuthenticationProvider(userService: UserService,
                                    userTokenService: UserTokenService,
                                    userLastActiveActorService: UserLastActiveActorService,
                                    userGroupService: UserGroupService,
                                    principalDataProvider: PrincipalDataProvider,
                                    personalAccessTokenFeature: PersonalAccessTokenFeature)
  extends AuthenticationProvider
    with Logging {

  private var authoritiesMapper: GrantedAuthoritiesMapper = new NullAuthoritiesMapper()

  @throws[AuthenticationException]
  override def authenticate(token: Authentication): Authentication = {
    logger.debug("Authenticating for Digital.ai Release")
    try {
      token match {
        case _: PersonalAuthenticationToken =>
          authenticateFromUserToken(token.getCredentials.toString)
        case _: OwnerAuthenticationToken =>
          authenticateFromReleaseOwner(token.getPrincipal.toString)
        case _ =>
          authenticateFromCredentials(token.getPrincipal.toString, token.getCredentials.toString)
      }
    } catch {
      case exc@(_: AuthenticationFailureException | _: NotFoundException) =>
        throw new BadCredentialsException(exc.getMessage, exc)
    }
  }

  private def authenticateFromUserToken(token: String): Authentication = {
    logger.trace("Authenticating using personal access token")
    val tokenHash = TokenGenerator.hash(token)
    val userToken = userTokenService.findByUserToken(tokenHash).getOrElse(
      throw new AuthenticationFailureException("Cannot authenticate with supplied personal access token")
    )
    if (userToken.isExpired()) {
      throw TokenExpiredException(s"The token expired on ${printDate(userToken.expiryDate)}")
    }
    userLastActiveActorService.updateTokenLastUsed(userToken.ciUid)
    val mappedAuthorities: util.Collection[_ <: GrantedAuthority] = evaluateAuthorities(userToken.username)
    new PersonalAuthenticationToken(new InternalUser(userToken.username),
      token,
      userToken.expiryDate,
      mappedAuthorities,
      Option(userToken.userTokenPermission).map(_.globalPermissions).getOrElse(Collections.emptySet()),
      Collections.emptySet(),
      Collections.emptySet())
  }

  private def authenticateFromCredentials(username: String, password: String): Authentication = {
    logger.trace(s"Authenticating [$username]")
    if (isBlank(username)) {
      throw new BadCredentialsException("Cannot authenticate with empty username")
    }
    userService.authenticate(username, password)
    val mappedAuthorities: util.Collection[_ <: GrantedAuthority] = evaluateAuthorities(username)
    new UsernamePasswordAuthenticationToken(new InternalUser(username), password, mappedAuthorities)
  }

  private def authenticateFromReleaseOwner(owner: String): Authentication = {
    logger.trace(s"Authenticating release owner: [$owner]")
    if (isBlank(owner)) {
      throw new BadCredentialsException("Cannot authenticate with empty release owner")
    }
    val mappedAuthorities: util.Collection[_ <: GrantedAuthority] = evaluateAuthorities(owner)
    new OwnerAuthenticationToken(new InternalUser(owner), mappedAuthorities)
  }

  private def evaluateAuthorities(username: String): util.Collection[_ <: GrantedAuthority] = {
    val userData = principalDataProvider.getUserData(username)
    if (userData.isFound) {
      // map authorities for LDAP, Crowd users
      authoritiesMapper.mapAuthorities(principalDataProvider.getAuthorities(username))
    } else {
      val authorities = try {
        // map authorities for internal users
        val user = userService.read(username)
        val grantedAuthorities = new util.ArrayList[GrantedAuthority]
        if (user.isAdmin || "admin" == user.getUsername) {
          grantedAuthorities.add(new SimpleGrantedAuthority(PermissionEnforcer.ROLE_ADMIN))
        }
        grantedAuthorities
      } catch {
        case _: NotFoundException if personalAccessTokenFeature.enabled =>
          // map authorities for SSO users like OIDC from saved group membership
          val userGroups = userGroupService.findGroupsForUser(username)
          val grantedAuthorities = new util.ArrayList[GrantedAuthority]
          userGroups.foreach { group =>
            grantedAuthorities.add(new SimpleGrantedAuthority(group))
          }
          grantedAuthorities
      }
      authoritiesMapper.mapAuthorities(authorities)
    }
  }

  override def supports(authentication: Class[_]): Boolean = {
    authentication.isAssignableFrom(classOf[UsernamePasswordAuthenticationToken]) ||
      authentication.isAssignableFrom(classOf[PersonalAuthenticationToken]) ||
      authentication.isAssignableFrom(classOf[OwnerAuthenticationToken])
  }

  def getAuthoritiesMapper: GrantedAuthoritiesMapper = authoritiesMapper

  def setAuthoritiesMapper(authoritiesMapper: GrantedAuthoritiesMapper): Unit = {
    this.authoritiesMapper = authoritiesMapper
  }

}
