package com.xebialabs.xlrelease.api.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import jakarta.servlet.MultipartConfigElement;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.Part;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.domain.TemplateLogo;
import com.xebialabs.xlrelease.export.TemplateImportContext;
import com.xebialabs.xlrelease.export.TemplateImporter;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.UploadService;
import com.xebialabs.xlrelease.views.AttachmentView;
import com.xebialabs.xlrelease.views.ImportResult;
import com.xebialabs.xlrelease.views.TemplateLogoView;

import static com.xebialabs.xlrelease.repository.Ids.isNullId;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;

/**
 * <p>
 * Handle file uploads used in various features of Digital.ai Release.
 * </p>
 * <p>
 * Upload URIs are not secured, because with IE the upload component falls back to an iframe, where we cannot pass authentication headers.
 * In order to limit the risk of exposing unsecured URIs, we make them temporary: before performing an upload, the client requests a
 * one-time token, that it must append to the URI it submits to.
 * </p>
 * Results must be TEXT_PLAIN because of IE9 iframe download handling, ZIP type makes IE believes it's a download of zip file.
 */
@Path("/upload")
@Consumes({MediaType.MULTIPART_FORM_DATA})
@Produces({MediaType.TEXT_HTML + "; charset=utf-8"})
@Controller
public class UploadResource {
    private static final String JSON_SUFFIX = ".json";

    private PermissionChecker permissionChecker;

    private TemplateImporter templateImporter;

    private UploadService uploadService;

    @Autowired
    public UploadResource(PermissionChecker permissionChecker, TemplateImporter templateImporter, UploadService uploadService) {
        this.permissionChecker = permissionChecker;
        this.templateImporter = templateImporter;
        this.uploadService = uploadService;
    }

    @SuppressWarnings("unused")
    public UploadResource() {
    }

    @POST
    @Timed
    @Path("templates/zip")
    public List<ImportResult> importTemplates(@Context HttpServletRequest request, @QueryParam("folderId") String folderId) throws IOException {
        if (isNullId(folderId)) {
            permissionChecker.check(XLReleasePermissions.CREATE_TEMPLATE);
        } else {
            permissionChecker.check(XLReleasePermissions.EDIT_TEMPLATE, folderId);
        }

        try {
            Optional<Part> templateInformation = request.getParts().stream().filter(this::filterFilePart).findFirst();
            if (templateInformation.isPresent()) {
                Part template = templateInformation.get();
                boolean isJson = template.getName().toLowerCase().endsWith(JSON_SUFFIX);
                boolean isDsl = template.getName().toLowerCase().endsWith(".zip");
                try (InputStream is = template.getInputStream()) {
                    return templateImporter.importTemplate(is, new TemplateImportContext(folderId, isJson,
                            false, isDsl, null));
                }
            } else {
                throw new IllegalArgumentException("Missing file");
            }
        } catch (ServletException e) {
            throw new BadRequestException("Expected multipart content");
        }
    }

    @POST
    @Timed
    @Path("attachment/{ciId}")
    public List<AttachmentView> addAttachment(@PathParam("ciId") @IdParam String ciId, @Context HttpServletRequest request) throws IOException {
        permissionChecker.checkEditAttachment(ciId);

        try {
            List<AttachmentView> returnedViews = this.uploadService.addAttachment(ciId, request.getParts().stream().filter((this::filterFilePart))
                    .iterator()).stream().map(AttachmentView::new).toList();
            if (returnedViews.isEmpty()) {
                throw new BadRequestException("Expected file upload");
            }
            return returnedViews;
        } catch (ServletException e) {
            throw new BadRequestException("Expected multipart content");
        } catch (IllegalStateException e) {
            throw new BadRequestException(e);
        }
    }

    @POST
    @Timed
    @Path("logo/{templateId:.*Release[^/-]*}")
    public TemplateLogoView addLogo(@PathParam("templateId") @IdParam String templateId, @Context HttpServletRequest request) throws IOException {
        permissionChecker.check(EDIT_TEMPLATE, templateId);

        request.setAttribute(org.eclipse.jetty.server.Request.__MULTIPART_CONFIG_ELEMENT, new MultipartConfigElement(System.getProperty("java.io.tmpdir"),
                TemplateLogo.getDefaultMaxLogoSize(), -1L, 0));
        try {
            int logoCount = request.getParts().stream().filter(this::filterFilePart).toList().size();
            if (logoCount > 1) {
                throw new BadRequestException("Can not upload more than one file");
            } else if (logoCount == 0) {
                throw new BadRequestException("No logo file provided");
            } else {
                return new TemplateLogoView(this.uploadService.addLogo(templateId,
                        request.getParts().stream().filter(this::filterFilePart).iterator()));
            }
        } catch (ServletException | IllegalArgumentException e) {
            throw new BadRequestException(e.getMessage());
        }
    }

    private Boolean filterFilePart(Part part) {
        return part.getSubmittedFileName() != null && !part.getSubmittedFileName().isEmpty();
    }
}
