package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.ext.Providers;

import com.xebialabs.xlrelease.domain.variables.ReferencedType;
import com.xebialabs.xlrelease.variable.DefaultCiValueProvider;
import org.jboss.resteasy.spi.ResteasyProviderFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlrelease.domain.Configuration;
import com.xebialabs.xlrelease.domain.ExportHook;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ConfigurationAutoconfigService;
import com.xebialabs.xlrelease.service.SharedConfigurationService;
import com.xebialabs.xlrelease.service.SharedConfigurationStatusService;
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;
import com.xebialabs.xlrelease.views.AutoconfigResponse;
import com.xebialabs.xlrelease.views.ConfigurationView;
import com.xebialabs.xlrelease.views.SharedConfigurationStatusResponse;
import com.xebialabs.xlrelease.views.converters.ConfigurationItemViewConverter;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.xlrelease.repository.IdType.CONFIGURATION;
import static org.jboss.resteasy.core.ResteasyContext.getContextDataMap;

/**
 * Configuration objects are used to externalize configuration that can be referenced by custom script tasks.
 * <p>
 * One can declare new configuration types by adding it in the synthetic file
 */
@Path("/configurations")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class SharedConfigurationResource {

    private SharedConfigurationService sharedConfigurationService;

    private PermissionChecker permissions;

    private ConfigurationItemViewConverter configurationViewConverter;

    private SharedConfigurationStatusService sharedConfigurationStatusService;
    private ConfigurationAutoconfigService configurationAutoconfigService;

    @Autowired
    public SharedConfigurationResource(SharedConfigurationService sharedConfigurationService, PermissionChecker permissions,
                                       ConfigurationItemViewConverter configurationViewConverter, SharedConfigurationStatusService sharedConfigurationStatusService,
                                       ConfigurationAutoconfigService configurationAutoconfigService) {
        this.sharedConfigurationService = sharedConfigurationService;
        this.permissions = permissions;
        this.sharedConfigurationStatusService = sharedConfigurationStatusService;
        this.configurationViewConverter = configurationViewConverter;
        this.configurationAutoconfigService = configurationAutoconfigService;
    }

    /**
     * Return the list of configuration types declared in synthetic file
     *
     * @return a list of type descriptor
     */
    @GET
    @Timed
    @Path("descriptors")
    public List<Descriptor> getDescriptors() {
        // Permission not checked here, data is required to resolve configuration instances on tasks
        return sharedConfigurationService.findAllConfigurationDescriptors();
    }

    /**
     * Return the list of configuration instance
     *
     * @param anonymized a boolean indicating whether the credentials should be stripped or not
     * @return a list of configuration instance possibly anonymized
     */
    @GET
    public List<ConfigurationView> getInstances(@QueryParam("anonymized") boolean anonymized) {
        if (anonymized) {
            // Permission not checked here, we don't send username/passwords.
            return sharedConfigurationService.findAll().stream().map(configurationViewConverter::toAnonymizedView).collect(Collectors.toList());
        } else {
            permissions.check(ADMIN);
            return sharedConfigurationService.findAll().stream().map(configurationViewConverter::toView).collect(Collectors.toList());
        }
    }

    /**
     * Creates a new configuration instance
     *
     * @param configurationView the information required to create the new instance
     * @return the created instance
     */
    @POST
    public ConfigurationView addInstance(ConfigurationView configurationView) {
        permissions.check(ADMIN);
        Configuration configuration = sharedConfigurationService.create(configurationViewConverter.fromView(configurationView));
        return configurationViewConverter.toView(configuration);
    }

    /**
     * Returns a configuration instance
     *
     * @param configurationInstanceId the configuration instance identifier
     * @return the configuration instance
     */
    @GET
    @Timed
    @Path("{configurationInstanceId:((?!descriptors).)*}")
    public ConfigurationView getInstance(@IdParam(CONFIGURATION) @PathParam("configurationInstanceId") String configurationInstanceId) {
        permissions.check(ADMIN);
        Configuration configuration = sharedConfigurationService.findById(configurationInstanceId);
        return configurationViewConverter.toView(configuration);
    }

    /**
     * Updates a configuration instance
     *
     * @param configurationInstanceId the configuration instance identifier
     * @param configurationView       the changes to apply
     * @return the updated configuration instance
     */
    @PUT
    @Timed
    @Path("{configurationInstanceId:((?!descriptors).)*}")
    public ConfigurationView updateInstance(@IdParam(CONFIGURATION) @PathParam("configurationInstanceId") String configurationInstanceId, ConfigurationView configurationView) {
        permissions.check(ADMIN);
        Configuration configuration = sharedConfigurationService.update(configurationInstanceId, configurationViewConverter.fromView(configurationView));
        return configurationViewConverter.toView(configuration);
    }

    /**
     * Deletes a configuration instance
     *
     * @param configurationInstanceId the configuration identifier
     */
    @DELETE
    @Timed
    @Path("{configurationInstanceId}")
    public void deleteInstance(@IdParam(CONFIGURATION) @PathParam("configurationInstanceId") String configurationInstanceId) {
        permissions.check(ADMIN);
        sharedConfigurationService.delete(configurationInstanceId);
    }

    @Timed
    @Path("status")
    @POST
    public SharedConfigurationStatusResponse checkConnection(ConfigurationView configurationView) {
        Configuration configuration = getConfiguration(configurationView);
        SharedConfigurationStatusResponse response = sharedConfigurationStatusService.checkStatus(configuration);
        fixRestEasyProviders();
        return response;
    }

    @Timed
    @Path("hasScript")
    @POST
    public Boolean hasScript(ConfigurationView configurationView) {
        Configuration configuration = configurationViewConverter.fromView(configurationView);
        return !(configuration instanceof ExportHook) && sharedConfigurationStatusService.hasScript(configuration);
    }


    @Timed
    @Path("autoconfigure")
    @POST
    public AutoconfigResponse autoconfigure(ConfigurationView configurationView) {
        Configuration configuration = getConfiguration(configurationView);
        AutoconfigResponse response = configurationAutoconfigService.autoconfigure(configuration);
        fixRestEasyProviders();
        return response;
    }

    @Timed
    @Path("isAutoconfigure")
    @POST
    public Boolean isAutoconfigure(ConfigurationView configurationView) {
        Configuration configuration = configurationViewConverter.fromView(configurationView);
        return configurationAutoconfigService.isAutoconfigure(configuration);
    }

    @Timed
    @Path("referenceVariable/types")
    @GET
    public List<ReferencedType> getReferenceVariableTypes() {
        return DefaultCiValueProvider.getAllSupportedTypes();
    }

    private Configuration getConfiguration(ConfigurationView configurationView) {
        Configuration configuration = configurationViewConverter.fromView(configurationView);

        String id = configuration.getId();

        Optional<ConfigurationItem> original = id == null ? Optional.empty() : Optional.of(sharedConfigurationService.findById(id));
        // Lookup previous version of the CI to replace any masked passwords with the actual value so the connection can be checked
        PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeededJava(original, configuration);
        return configuration;
    }

    private void fixRestEasyProviders() {
        // REL-2644
        if (getContextDataMap().isEmpty()) {
            getContextDataMap().put(Providers.class, ResteasyProviderFactory.getInstance());
        }
    }
}
