package com.xebialabs.xlrelease.api.internal;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.core.rest.api.DtoReader;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.api.v1.filters.RolePrincipalsFilters;
import com.xebialabs.xlrelease.domain.events.GlobalRolesOrPermissionsUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.ReleaseRoleService;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.utils.RolePrincipalEventPublisher;
import com.xebialabs.xlrelease.views.RolePrincipalsView;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.api.ApiService.DEFAULT_RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;
import static java.util.stream.Collectors.toList;

/**
 * The login names or LDAP groups that are part of a security role.
 */
@Path("/roles")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class RolePrincipalsResource {

    private static final Logger logger = LoggerFactory.getLogger(RolePrincipalsResource.class);

    private final PermissionChecker permissionChecker;
    private final RoleService roleService;
    private final TeamService teamService;
    private final ReleaseRoleService releaseRoleService;
    private final XLReleaseEventBus xlReleaseEventBus;
    private final DtoReader dtoReader = new DtoReader();

    @Autowired
    public RolePrincipalsResource(PermissionChecker permissionChecker,
                                  RoleService roleService,
                                  TeamService teamService,
                                  ReleaseRoleService releaseRoleService,
                                  XLReleaseEventBus xlReleaseEventBus) {
        this.permissionChecker = permissionChecker;
        this.roleService = roleService;
        this.teamService = teamService;
        this.releaseRoleService = releaseRoleService;
        this.xlReleaseEventBus = xlReleaseEventBus;
    }

    @GET
    @Timed
    @Path("names")
    public List<String> readRoleNames() {
        return this.roleService.getRoles().stream().map(Role::getName).collect(toList());
    }

    @GET
    @Timed
    @Path("principals")
    public Page<RolePrincipalsView> readRolePrincipals(
            @BeanParam RolePrincipalsFilters filters,
            @DefaultValue("0") @QueryParam(PAGE) int page,
            @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE);
        PageRequest pageable = PageRequest.of(page, resultsPerPage);
        return this.readRolePrincipals(filters, pageable);
    }

    // Used by as-code
    @Timed
    @Deprecated(since = "24.1")
    public List<RolePrincipalsView> readRolePrincipals() {
        var filters = new RolePrincipalsFilters(null, null);
        Page<RolePrincipalsView> results = this.readRolePrincipals(filters,  Pageable.unpaged());
        return results.getContent();
    }

    @PUT
    @Timed
    @Path("principals")
    public void createOrUpdateGlobalRolePermission(final RolePrincipalsView rolePrincipalsView) {
        permissionChecker.check(EDIT_SECURITY);
        RolePrincipals rolePrincipals = rolePrincipalsView.toRolePrincipals();
        teamService.generateIdIfNecessary(rolePrincipals.getRole());
        Role role = dtoReader.readRoleAssignments(Collections.singletonList(rolePrincipals)).get(0);
        logger.debug(String.format("createOrUpdateGlobalRolePermission: role := %s / %s", role.getId(), role.getName()));

        Optional<Role> originalRole = getRole(role.getId());

        roleService.createOrUpdateRole(role);
        RolePrincipalEventPublisher.publishCreateOrUpdate(originalRole, role);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
    }

    @DELETE
    @Timed
    @Path("principals/{roleId:.*}")
    public void deleteGlobalRolePermission(@PathParam("roleId") String roleId) {
        permissionChecker.check(EDIT_SECURITY);
        Optional<Role> role = getRole(roleId);
        if (role.isPresent()) {
            roleService.deleteById(roleId);
            RolePrincipalEventPublisher.publishDelete(role);
            xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
        }
    }

    private Page<RolePrincipalsView> readRolePrincipals(RolePrincipalsFilters filters, Pageable pageable) {
        permissionChecker.check(EDIT_SECURITY);
        return releaseRoleService.getGlobalRolePrincipalViews(filters, pageable);
    }

    private Optional<Role> getRole(final String roleId) {
        try {
            Role role = roleService.getRoleForRoleId(roleId);
            return Optional.of(role);
        } catch (NotFoundException nfe) {
            logger.trace(nfe.getMessage());
            return Optional.empty();
        }
    }
}
