package com.xebialabs.xlrelease.api.internal;

import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.views.DatesForm;
import com.xebialabs.xlrelease.views.LinkForm;
import com.xebialabs.xlrelease.views.LinkView;

import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.views.converters.DurationViewConverter.fromDurationView;

/**
 * The GANTT planner used to adjust a release's dates and links.
 */
@Path("/planning")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class PlanningResource {

    private PermissionChecker permissionChecker;
    private ReleaseActorService releaseActorService;

    @Autowired
    public PlanningResource(PermissionChecker permissionChecker, ReleaseActorService releaseActorService) {
        this.releaseActorService = releaseActorService;
        this.permissionChecker = permissionChecker;
    }

    @PUT
    @Timed
    @Path("{planItemId:.*(Release|Phase|Task)[^/-]*}")
    public void updateDates(@PathParam("planItemId") @IdParam String planItemId, DatesForm datesForm) {
        permissionChecker.checkEditDate(planItemId);
        releaseActorService.updateDates(planItemId, datesForm.getScheduledStartDate(), datesForm.getDueDate(), fromDurationView(datesForm.getPlannedDuration()));
    }

    @POST
    @Timed
    @Path("links/{containerId:.*Task[^/-]*}")
    public LinkView addLink(@PathParam("containerId") @IdParam String containerId, LinkForm linkForm) {
        permissionChecker.checkEdit(releaseIdFrom(containerId));
        return new LinkView(releaseActorService.createLink(containerId, linkForm.getSourceId(), linkForm.getTargetId()));
    }

    @DELETE
    @Timed
    @Path("links/{linkId:.*Link[^/-]*}")
    public void removeLink(@PathParam("linkId") @IdParam String linkId) {
        permissionChecker.checkEdit(releaseIdFrom(linkId));
        releaseActorService.deleteLink(linkId);
    }
}
