package com.xebialabs.xlrelease.api.internal;

import com.codahale.metrics.annotation.Timed;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Dependency;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.features.TaskUpdateVersioningFeature;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.TaskGranularPermissions;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.DependencyService;
import com.xebialabs.xlrelease.service.TaskService;
import com.xebialabs.xlrelease.views.DependencyNode;
import com.xebialabs.xlrelease.views.DependencyView;
import com.xebialabs.xlrelease.views.ReleaseDependenciesView;
import com.xebialabs.xlrelease.views.ReleaseTree;
import com.xebialabs.xlrelease.views.converters.DependencyNodeConverter;
import com.xebialabs.xlrelease.views.converters.DependencyViewConverter;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import java.util.List;

import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static java.util.stream.Collectors.toList;

/**
 * Dependencies from a gate to another release (possibly narrowed down to a phase and a task).
 */
@Path("/dependencies")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class DependencyResource {

    private PermissionChecker permissions;

    private DependencyService dependencyService;

    private DependencyNodeConverter dependencyNodeConverter;

    private ReleaseActorService releaseActorService;

    private TaskGranularPermissions taskPermissionChecker;

    private TaskUpdateVersioningFeature taskUpdateVersioningFeature;

    private TaskService taskService;

    @Autowired
    public DependencyResource(PermissionChecker permissions,
                              DependencyService dependencyService,
                              DependencyNodeConverter dependencyNodeConverter,
                              ReleaseActorService releaseActorService,
                              TaskGranularPermissions taskPermissionChecker,
                              TaskUpdateVersioningFeature taskUpdateVersioningFeature,
                              TaskService taskService) {
        this.permissions = permissions;
        this.dependencyService = dependencyService;
        this.dependencyNodeConverter = dependencyNodeConverter;
        this.releaseActorService = releaseActorService;
        this.taskPermissionChecker = taskPermissionChecker;
        this.taskUpdateVersioningFeature = taskUpdateVersioningFeature;
        this.taskService = taskService;
    }

    @SuppressWarnings("unused")
    public DependencyResource() {
    }

    @PUT
    @Timed
    @Path("{dependencyId:.*Dependency[^/-]*}")
    public DependencyView updateDependency(@PathParam("dependencyId") @IdParam String dependencyId, @QueryParam("modifiedAt") Long modifiedAt, DependencyView dependencyView) {
        String releaseId = releaseIdFrom(dependencyId);
        Dependency dependency = getDirectivesAndUpdate(releaseId, dependencyId, dependencyView, modifiedAt);
        DependencyView updatedDependencyView = DependencyViewConverter.toDependencyView(dependency);

        updateModifiedDetails(updatedDependencyView, Ids.getParentId(dependencyId));
        return updatedDependencyView;
    }

    private Dependency getDirectivesAndUpdate(final String releaseId, final String dependencyId, final DependencyView dependencyView, final Long modifiedAt) {
        Dependency dependency = new Dependency();
        if (taskPermissionChecker.hasDirectiveToUpdateAllProperties(releaseId) ||
                taskPermissionChecker.hasEditTaskConfigurationGranularPermission(releaseId)) {
            checkViewPermissionIfNecessary(dependencyView);
            if (isTaskConcurrencyCheckEnabled()) {
                dependency = releaseActorService.updateDependencyVersioned(releaseId, dependencyId, dependencyView.toTargetIdOrVariable(), getModifiedAtDate(modifiedAt));
            } else {
                dependency = releaseActorService.updateDependency(releaseId, dependencyId, dependencyView.toTargetIdOrVariable());
            }
        }
        return dependency;
    }

    @DELETE
    @Timed
    @Path("{dependencyId:.*Dependency[^/-]*}")
    public void deleteDependency(@PathParam("dependencyId") @IdParam String dependencyId) {
        String releaseId = releaseIdFrom(dependencyId);
        permissions.checkEditTask(releaseId);

        releaseActorService.deleteDependency(releaseId, dependencyId);
    }

    @GET
    @Timed
    @Path("{releaseId:.*Release[^/-]*}")
    public ReleaseDependenciesView getReleaseDependencies(
            @PathParam("releaseId") @IdParam String releaseId,
            @QueryParam(value = "asReleases") boolean asReleases,
            @QueryParam("properties") List<String> properties,
            @QueryParam("extensions") List<String> extensions
    ) {
        permissions.checkView(releaseId);

        List<String> activeIncomingGateIds = dependencyService.findActiveIncomingGateIds(releaseId);
        List<String> activeOutgoingTargetIds = dependencyService.findActiveOutgoingTargetIds(releaseId);

        if (asReleases) {
            activeIncomingGateIds = toReleaseIds(activeIncomingGateIds);
            activeOutgoingTargetIds = toReleaseIds(activeOutgoingTargetIds);
        }

        List<DependencyNode> incomingDependenciesView = dependencyNodeConverter.toDependencyNodes(activeIncomingGateIds, properties, extensions);
        List<DependencyNode> outgoingDependenciesView = dependencyNodeConverter.toDependencyNodes(activeOutgoingTargetIds, properties, extensions);

        return new ReleaseDependenciesView(releaseId, incomingDependenciesView, outgoingDependenciesView);
    }

    @GET
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/tree")
    public ReleaseTree getReleaseTree(@PathParam("releaseId") @IdParam String releaseId) {
        permissions.checkView(releaseId);

        return dependencyService.getReleaseTree(releaseId);
    }

    private List<String> toReleaseIds(final List<String> targetId) {
        return targetId.stream().map(Ids::releaseIdFrom).distinct().collect(toList());
    }

    private void checkViewPermissionIfNecessary(final DependencyView dependencyView) {
        if (dependencyView.hasFixedTarget()) {
            permissions.checkView(releaseIdFrom(dependencyView.toTargetId()));
        }
    }

    private boolean isTaskConcurrencyCheckEnabled() {
        return taskUpdateVersioningFeature.concurrentTaskModificationProtectionEnabled();
    }

    private DateTime getModifiedAtDate(Long modifiedAt) {
        if (modifiedAt != null) {
            return new DateTime(modifiedAt);
        }
        return null;
    }

    private void updateModifiedDetails(DependencyView dependencyView, String taskId) {
        Task updatedTask = taskService.findById(taskId, ResolveOptions.WITHOUT_DECORATORS());
        CiAttributes ciAttributes = updatedTask.get$ciAttributes();
        dependencyView.setLastModifiedBy(ciAttributes.getLastModifiedBy());
        if (updatedTask.get$ciAttributes().getLastModifiedAt() == null) {
            dependencyView.setLastModifiedAt(null);
        } else {
            dependencyView.setLastModifiedAt(ciAttributes.getLastModifiedAt().toDate());
        }
    }
}
