package com.xebialabs.xlrelease.service;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.xlrelease.domain.Task;
//import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.repository.UserProfileRepository;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ConcurrentModificationException;
import java.util.Set;

import static org.joda.time.DateTimeConstants.*;

@Service
public class TaskConcurrencyService {

    private final UserProfileRepository userProfileRepository;

    @Autowired
    public TaskConcurrencyService(UserProfileRepository userProfileRepository){
        this.userProfileRepository = userProfileRepository;
    }

    /**
     * This method will update the lastModifiedBy field to the current authorized user and will update the lastModifiedAt
     * field to the current time in UTC
     *
     * @param task - Task that will be updated
     */
    public void updateLastModifiedDetails(Task task) {
        task.set$ciAttributes(new CiAttributes(task.get$ciAttributes().getCreatedBy(),
                task.get$ciAttributes().getCreatedAt(), Permissions.getAuthenticatedUserName(),
                DateTime.now().withZone(DateTimeZone.UTC), task.get$ciAttributes().getScmTraceabilityDataId()));
    }

    /**
     * This method will compare a task to be updated with the existing task when the update directive
     * UPDATE_VERIFY_CONCURRENT_MODIFICATION is specified to determine if the tasks version has been modified after
     * the updated task. The version check will fail if the user who last modified the existingTask is different from
     * the requested user and the existingTask has a modifiedAt time after the modifiedAt time of the updatedTask.
     *
     * @param existingTask     - Existing task
     * @param updatedTask      - Updated task with requested changes
     * @param updateDirectives - Update directives included in request
     */
    public void checkConcurrentModification(final Task existingTask, final Task updatedTask,
                                            Set<TaskUpdateDirective> updateDirectives) {
        if (updateDirectives.contains(TaskUpdateDirective.UPDATE_VERIFY_CONCURRENT_MODIFICATION)) {
            CiAttributes existingTaskAttributes = existingTask.get$ciAttributes();
            CiAttributes updatedTaskAttributes = updatedTask.get$ciAttributes();
            if (isLastUpdatedByDifferentUser(existingTaskAttributes.getLastModifiedBy()) &&
                    isUpdatedTaskVersionOutdated(existingTaskAttributes.getLastModifiedAt(), updatedTaskAttributes.getLastModifiedAt())) {
                throw new ConcurrentModificationException(formatExceptionMessage(existingTaskAttributes));
            }
        }
    }

    /**
     * This method will compare a task to be updated with the provided modifiedAt time to determine if the tasks version
     * has been modified after the updated task request. The version check will fail if the user who last modified the
     * existing is different from the requested user and the existingTask has a modifiedAt time after the provided
     * time.
     *
     * @param existingTask - Existing task
     * @param modifiedAt   - Task last modified at time reported by client
     */
    public void checkConcurrentModification(final Task existingTask, DateTime modifiedAt) {
        CiAttributes existingTaskAttributes = existingTask.get$ciAttributes();
        if (isLastUpdatedByDifferentUser(existingTaskAttributes.getLastModifiedBy()) &&
                isUpdatedTaskVersionOutdated(existingTaskAttributes.getLastModifiedAt(), modifiedAt)) {
            throw new ConcurrentModificationException(formatExceptionMessage(existingTaskAttributes));
        }
    }

    private boolean isLastUpdatedByDifferentUser(String originalModifiedBy) {
        if (originalModifiedBy == null) {
            return true;
        }
        return originalModifiedBy.compareToIgnoreCase(Permissions.getAuthenticatedUserName()) != 0;
    }

    private boolean isUpdatedTaskVersionOutdated(DateTime originalModDate, DateTime updatedModDate) {
        return (originalModDate != null && updatedModDate != null &&
                originalModDate.isAfter(updatedModDate.withZone(DateTimeZone.UTC))) ||
                (originalModDate != null && updatedModDate == null);
    }

    private String formatExceptionMessage(CiAttributes ciAttributes) {
        return String.format("This task has been modified by %s %s ago. Refresh the task to see the most " +
                        "recent version and apply the changes.",
                getLastModifiedByUser(ciAttributes.getLastModifiedBy()),
                formatTimeSinceLastChange(ciAttributes.getLastModifiedAt().withZone(DateTimeZone.UTC)));
    }

    private String getLastModifiedByUser(String lastModifiedBy) {
        if (lastModifiedBy == null) {
            return "Unknown";
        } else {
            UserProfile userProfile = userProfileRepository.findById(lastModifiedBy.toLowerCase()).getOrElse(() -> null);
            return userProfile != null && userProfile.getFullName() != null && !userProfile.getFullName().isEmpty()
                    ? userProfile.getFullName() : lastModifiedBy;
        }
    }

    private String formatTimeSinceLastChange(DateTime dateTime) {
        long timeSinceLastUpdate = DateTime.now().withZone(DateTimeZone.UTC).getMillis() - dateTime.getMillis();

        if (timeSinceLastUpdate < MILLIS_PER_MINUTE) {
            return String.format("%d second(s)", timeSinceLastUpdate / MILLIS_PER_SECOND);
        } else if (timeSinceLastUpdate < MILLIS_PER_HOUR) {
            return String.format("%d minute(s)", timeSinceLastUpdate / MILLIS_PER_MINUTE);
        } else if (timeSinceLastUpdate < MILLIS_PER_DAY) {
            return String.format("%d hour(s)", timeSinceLastUpdate / MILLIS_PER_HOUR);
        } else {
            return String.format("%d day(s)", timeSinceLastUpdate / MILLIS_PER_DAY);
        }
    }
}
