package com.xebialabs.xlrelease.service;

import com.codahale.metrics.annotation.Timed;
import com.xebialabs.xlrelease.domain.GateCondition;
import com.xebialabs.xlrelease.domain.GateTask;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.events.GateConditionCreatedEvent;
import com.xebialabs.xlrelease.domain.events.GateConditionDeletedEvent;
import com.xebialabs.xlrelease.domain.events.GateConditionUpdatedEvent;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.GateConditionRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.TaskRepository;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables;

@Service
public class GateConditionService {

    private final static String DEFAULT_TITLE = "New Condition";

    private CiIdService ciIdService;

    private GateConditionRepository gateConditionRepository;

    private TaskConcurrencyService taskConcurrencyService;

    private XLReleaseEventBus eventBus;

    private TaskRepository taskRepository;

    @Autowired
    public GateConditionService(CiIdService ciIdService,
                                XLReleaseEventBus eventBus,
                                TaskRepository taskRepository,
                                GateConditionRepository gateConditionRepository,
                                TaskConcurrencyService taskConcurrencyService) {
        this.ciIdService = ciIdService;
        this.eventBus = eventBus;
        this.gateConditionRepository = gateConditionRepository;
        this.taskRepository = taskRepository;
        this.taskConcurrencyService = taskConcurrencyService;
    }

    @Timed
    public GateCondition create(String parentId) {
        return create(parentId, DEFAULT_TITLE);
    }

    @Timed
    public GateCondition create(String gateId, String text) {
        GateTask gate = taskRepository.findById(gateId);

        LockedTaskOperationChecks.checkCreateGateCondition(gate);
        checkArgument(gate.isUpdatable(), "You can't add a condition to a finished gate");

        GateCondition condition = new GateCondition();
        condition.setId(ciIdService.getUniqueId(condition.getType(), gateId));
        condition.setTitle(text);
        gate.addCondition(condition);

        List<Variable> variables = updateVariables(gate);

        gateConditionRepository.create(condition, variables);

        eventBus.publish(new GateConditionCreatedEvent(gate, condition));

        return condition;
    }

    @Timed
    public GateCondition findById(String conditionId) {
        return gateConditionRepository.findById(conditionId);
    }

    @Timed
    public GateCondition update(String conditionId, GateCondition updatedCondition, DateTime modifiedAt) {
        String gateId = Ids.getParentId(updatedCondition.getId());
        GateTask gate = taskRepository.findById(gateId);
        taskConcurrencyService.checkConcurrentModification(gate, modifiedAt);
        return update(conditionId, updatedCondition, gate);
    }

    @Timed
    public GateCondition update(String conditionId, GateCondition updatedCondition) {
        String gateId = Ids.getParentId(updatedCondition.getId());
        GateTask gate = taskRepository.findById(gateId);
        return update(conditionId, updatedCondition, gate);
    }

    @Timed
    public void delete(String conditionId) {
        String gateId = Ids.getParentId(conditionId);
        GateTask gate = taskRepository.findById(gateId);
        LockedTaskOperationChecks.checkDeleteGateCondition(gate);
        checkArgument(gate.isUpdatable(), "You can't delete a condition from a finished gate");

        GateCondition gateCondition = findById(conditionId);
        gateConditionRepository.delete(gateCondition);

        eventBus.publish(new GateConditionDeletedEvent(gate, gateCondition));
    }

    private List<Variable> updateVariables(final GateTask gate) {
        Release release = gate.getRelease();
        return scanAndBuildNewVariables(release, gate, ciIdService);
    }

    private GateCondition update(String conditionId, GateCondition updatedCondition, GateTask gate) {
        checkArgument(gate.isUpdatable(), "You can't update a condition of a finished gate.");
        GateCondition condition = gate.getCondition(conditionId);
        LockedTaskOperationChecks.checkUpdateGateCondition(gate, condition, updatedCondition);
        GateCondition originalCondition = cloneCi(condition);
        condition.setChecked(updatedCondition.isChecked());
        condition.setTitle(updatedCondition.getTitle());

        List<Variable> variables = updateVariables(gate);
        gateConditionRepository.update(updatedCondition, variables);

        eventBus.publish(new GateConditionUpdatedEvent(gate, originalCondition, updatedCondition));
        return updatedCondition;
    }
}
