package com.xebialabs.xlrelease.api.internal;

import java.util.*;
import java.util.stream.Collectors;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.rest.api.ServerResource;
import com.xebialabs.deployit.plumbing.CurrentVersion;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.authentication.viewas.ViewAsAuthenticationFinder;
import com.xebialabs.deployit.security.authentication.viewas.ViewAsData;
import com.xebialabs.license.service.LicenseService;
import com.xebialabs.plugin.manager.service.PluginService;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.support.report.repository.DataStatisticsRepository;
import com.xebialabs.xlrelease.views.ClusterInfoView;
import com.xebialabs.xlrelease.views.LicenseView;
import com.xebialabs.xlrelease.views.PluginView;
import com.xebialabs.xlrelease.views.ServerInfoView;

import static com.xebialabs.deployit.ServerConfiguration.KEY_CLIENT_SESSION_REMEMBER_ENABLED;
import static com.xebialabs.xlrelease.api.utils.AccountIdHelper.generateAccountId;
import static scala.jdk.javaapi.CollectionConverters.asJavaCollection;

/**
 * The server running XL Release.
 */
@Path("/server")
@Controller
public class ReleaseServerResource extends ServerResource {

    private static final Logger logger = LoggerFactory.getLogger(ReleaseServerResource.class);

    private final ServerConfiguration serverConfiguration;
    private final LicenseService licenseService;
    private final PluginService pluginService;
    private final DataStatisticsRepository dataStatisticsRepository;

    @Autowired
    public ReleaseServerResource(LicenseService licenseService,
                                 List<ViewAsAuthenticationFinder> viewAsAuthenticationFinders,
                                 RoleService roleService,
                                 ServerConfiguration serverConfiguration,
                                 ViewAsData viewAsData,
                                 PluginService pluginService,
                                 DataStatisticsRepository dataStatisticsRepository) {
        super(licenseService, viewAsAuthenticationFinders, roleService, serverConfiguration, viewAsData);
        this.serverConfiguration = serverConfiguration;
        this.licenseService = licenseService;
        this.pluginService = pluginService;
        this.dataStatisticsRepository = dataStatisticsRepository;
    }

    @GET
    @Timed
    @Path("host")
    @Produces(MediaType.APPLICATION_JSON)
    public ServerInfoView getHostUrl() {
        return new ServerInfoView(serverConfiguration.getServerUrl());
    }

    @GET
    @Timed
    @Path("license")
    @Produces(MediaType.APPLICATION_JSON)
    public LicenseView getLicence() {
        return new LicenseView(licenseService.getLicense());
    }

    @GET
    @Timed
    @Path("cluster")
    @Produces(MediaType.APPLICATION_JSON)
    public ClusterInfoView getClusterInfo() {
        ClusterInfoView clusterInfoView = new ClusterInfoView();
        clusterInfoView.setClusterMode(XlrConfig.getInstance().clusterMode().configOption());
        clusterInfoView.setClusterName(XlrConfig.getInstance().cluster().name());
        if (XlrConfig.getInstance().cluster().config().hasPath("clusterPort")) {
            clusterInfoView.setClusterPort(XlrConfig.getInstance().cluster().config().getInt("clusterPort"));
        }
        if (XlrConfig.getInstance().cluster().config().hasPath("hostname")) {
            clusterInfoView.setNodeHostname(XlrConfig.getInstance().cluster().config().getString("hostname"));
        }
        return clusterInfoView;
    }

    @GET
    @Timed
    @Path("configuration")
    @Produces(MediaType.APPLICATION_JSON)
    public Map<String, Object> getProperties() {
        Map<String, Object> properties = new HashMap<>();
        properties.put(KEY_CLIENT_SESSION_REMEMBER_ENABLED, serverConfiguration.isClientSessionRememberEnabled());

        return properties;
    }

    @GET
    @Timed
    @Path("account-information")
    @Produces(MediaType.APPLICATION_JSON)
    public Map<String, Object> getAccountInformation() {
        Map<String, Object> properties = new HashMap<>();

        try {
            LicenseView licenseView = new LicenseView(licenseService.getLicense());
            Collection<?> installedPlugins = asJavaCollection(pluginService.extend(pluginService.listInstalled())
                    .map(dto -> new PluginView(dto.plugin())));

            properties.put("serverUrl", serverConfiguration.getServerUrl());
            properties.put("name", licenseView.getLicensedTo());
            properties.put("product", licenseView.getProduct());
            properties.put("edition", licenseView.getEdition());
            properties.put("version", CurrentVersion.get());
            properties.put("plugins", installedPlugins.stream().map(Object::toString).collect(Collectors.toList()));
            properties.put("accountId", generateAccountId(licenseView.getLicensedTo(), serverConfiguration.getServerUrl()));

            String dbName = "unknown";
            try {
                dbName = dataStatisticsRepository.findMainDbImplementation();
            } finally {
                properties.put("database", dbName);
            }
        } catch (Exception exception) {
            logger.warn("Unable to fetch account information", exception);
        }

        return properties;
    }


}
