package com.xebialabs.xlrelease.service


import com.xebialabs.xlrelease.api.v1.views.TeamMemberView
import com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType
import com.xebialabs.xlrelease.domain.TeamPermissionOperations
import com.xebialabs.xlrelease.views.TeamView

import scala.jdk.CollectionConverters._


sealed trait TeamUpdateOperation extends TeamPermissionOperations {
  def containerId: String

  def isEmpty: Boolean = false
}

case class TeamMembership(principals: Set[String] = Set.empty, roles: Set[String] = Set.empty, permissions: Set[String] = Set.empty) {

  def isEmpty: Boolean = principals.isEmpty && roles.isEmpty && permissions.isEmpty

  def copyWith(principal: Option[String] = None, role: Option[String] = None, permission: Option[String] = None): TeamMembership = {
    TeamMembership(
      this.principals ++ principal,
      this.roles ++ role,
      this.permissions ++ permission
    )
  }

  def copyWithout(principal: Option[String] = None, role: Option[String] = None, permission: Option[String] = None): TeamMembership = {
    TeamMembership(
      this.principals -- principal,
      this.roles -- role,
      this.permissions -- permission
    )
  }

  private def containsIfNeeds(needs: Option[String], values: Set[String]): Boolean = {
    needs.map(value => values.contains(value)).getOrElse(true)
  }

  def contains(principal: Option[String] = None, role: Option[String] = None, permission: Option[String] = None): Boolean = {
    containsIfNeeds(principal, this.principals) &&
      containsIfNeeds(role, this.roles) &&
      containsIfNeeds(permission, this.permissions)
  }
}

object TeamMembership {
  def apply(teamView: TeamView): TeamMembership = {
    val members = teamView.getMembers.asScala.toSet
    val membersPerType: Map[MemberType, Set[String]] = members.groupMap(m => m.getType)(m => m.getName)
    val roles = membersPerType.getOrElse(TeamMemberView.MemberType.ROLE, Set())
    val principals = membersPerType.getOrElse(TeamMemberView.MemberType.PRINCIPAL, Set())
    val teamPermissions = teamView.getPermissions.asScala.toSet
    this (principals, roles, teamPermissions)
  }

}

case class CreateTeamOperation(containerId: String, team: String, membership: TeamMembership = TeamMembership()) extends TeamUpdateOperation {
  override def getTeamName: String = team

  override def hasPermission(permission: String): Boolean = membership.permissions.contains(permission)
}

case class UpdateTeamOperation(containerId: String,
                               teamId: Option[String],
                               team: String,
                               addedMembership: TeamMembership = TeamMembership(),
                               removedMembership: TeamMembership = TeamMembership()) extends TeamUpdateOperation {
  override def getTeamName: String = team

  override def hasPermission(permission: String): Boolean = addedMembership.permissions.contains(permission)

  override def isEmpty: Boolean = addedMembership.isEmpty && removedMembership.isEmpty
}

case class DeleteTeamOperation(containerId: String, teamId: Option[String] = None, team: String) extends TeamUpdateOperation {
  override def getTeamName: String = team

  override def hasPermission(permission: String): Boolean = false
}

case class BreakInheritanceOperation(containerId: String) extends TeamUpdateOperation {
  override def getTeamName: String = null

  override def hasPermission(permission: String): Boolean = false
}
