package com.xebialabs.xlrelease.api.v1.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.api.v1.FolderApi;
import com.xebialabs.xlrelease.api.v1.TeamFacade;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.ReleaseKind;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Page;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.ExternalVariableService;
import com.xebialabs.xlrelease.service.FolderService;
import com.xebialabs.xlrelease.service.FolderVariableService;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.domain.folder.Folder.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.repository.Ids.isRoot;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.AUDIT_ALL;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.CREATE_TOP_LEVEL_FOLDER;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_SECURITY;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_TEAMS;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_FOLDER;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_TEMPLATE;


@Controller
@SuppressWarnings("RSReferenceInspection")
public class FolderApiImpl implements FolderApi {

    private final PermissionChecker permissions;
    private final FolderService folderService;
    private final TeamFacade teamFacade;
    private final FolderVariableService folderVariableService;
    private final ExternalVariableService externalVariableService;

    @Autowired
    public FolderApiImpl(PermissionChecker permissions,
                         FolderService folderService,
                         TeamFacade teamFacade,
                         FolderVariableService folderVariableService,
                         ExternalVariableService externalVariableService) {
        this.permissions = permissions;
        this.folderService = folderService;
        this.teamFacade = teamFacade;
        this.folderVariableService = folderVariableService;
        this.externalVariableService = externalVariableService;
    }

    @Timed
    @Override
    public List<Folder> listRoot(Long page, Long resultPerPage, Integer depth, Boolean decorateWithPermissions) {
        return folderService.listViewableFolders(ROOT_FOLDER_ID,
                Page.parse(Optional.ofNullable(page), Optional.ofNullable(resultPerPage), Optional.ofNullable(depth)),
                Optional.ofNullable(decorateWithPermissions).orElse(false), true);
    }

    @Timed
    @Override
    public List<Folder> list(String parentId, Long page, Long resultPerPage, Integer depth, Boolean decorateWithPermissions) {
        return folderService.listViewableFolders(parentId,
                Page.parse(Optional.ofNullable(page), Optional.ofNullable(resultPerPage), Optional.ofNullable(depth)),
                Optional.ofNullable(decorateWithPermissions).orElse(true), true);
    }

    @Timed
    @Override
    public Folder find(String path, Integer depth) {
        Folder byPath = folderService.findByPath(path, Optional.ofNullable(depth).orElse(Page.DEFAULT_DEPTH()));
        if (!permissions.hasGlobalPermission(AUDIT_ALL)) {
            permissions.check(VIEW_FOLDER, byPath.getId());
        }
        return byPath;
    }

    @Timed
    @Override
    public Folder getFolder(String folderId, Integer depth) {
        if (!permissions.hasGlobalPermission(AUDIT_ALL)) {
            permissions.check(VIEW_FOLDER, folderId);
        }
        return folderService.findViewableFoldersById(folderId, Optional.ofNullable(depth).orElse(Page.DEFAULT_DEPTH()), true);
    }

    @Timed
    @Override
    public Folder getFolder(String folderId) {
        return getFolder(folderId, null);
    }

    @Timed
    @Override
    public List<Release> getTemplates(String folderId, String kind, Long page, Long resultPerPage, Integer depth) {
        permissions.checkViewFolder(folderId);
        if (!isRoot(folderId) && !permissions.hasGlobalPermission(AUDIT_ALL)) {
            permissions.check(VIEW_TEMPLATE, folderId);
        }
        return folderService.searchTemplates(folderId, ReleaseKind.fromString(kind),
                Page.parse(Optional.ofNullable(page), Optional.ofNullable(resultPerPage), Optional.ofNullable(depth)), true);
    }

    @Timed
    @Override
    public List<Release> getTemplates(String folderId) {
        return getTemplates(folderId, null, null, null);
    }

    @Timed
    @Override
    public List<Release> searchReleases(String folderId, Long page, Long resultPerPage, Integer depth, ReleasesFilters releasesFilters) {
        if (!isRoot(folderId)) {
            permissions.checkViewFolder(folderId);
            permissions.canViewRelease(folderId);
        }
        if (null == releasesFilters) {
            releasesFilters = new ReleasesFilters();
        }
        return folderService.searchReleases(folderId, releasesFilters,
                Page.parse(Optional.ofNullable(page), Optional.ofNullable(resultPerPage), Optional.ofNullable(depth))).getReleases();
    }

    @Timed
    @Override
    public List<Release> searchReleases(String folderId, ReleasesFilters releasesFilters) {
        return searchReleases(folderId, null, null, null, releasesFilters);
    }

    @Timed
    @Override
    public void moveTemplate(String folderId, String templateId, Boolean mergeTeams) {
        permissions.checkEdit(templateId);
        permissions.check(EDIT_TEMPLATE, folderId);
        boolean shouldMergeTeams = Optional.ofNullable(mergeTeams).orElse(false);
        if (shouldMergeTeams) {
            permissions.check(EDIT_FOLDER_SECURITY, folderId);
        }
        folderService.moveTemplate(folderId, templateId, shouldMergeTeams);
    }

    @Timed
    @Override
    public void moveTemplate(String folderId, String templateId) {
        moveTemplate(folderId, templateId, null);
    }

    @Timed
    @Override
    public Folder addFolder(String parentId, Folder folder) {
        if (ROOT_FOLDER_ID.equals(parentId)) {
            permissions.check(CREATE_TOP_LEVEL_FOLDER);
        } else {
            permissions.check(EDIT_FOLDER, parentId);
        }
        return folderService.create(parentId, folder);
    }

    @Timed
    @Override
    public void delete(String folderId) {
        checkArgument(!ROOT_FOLDER_ID.equals(folderId), "Cannot delete root folder");
        permissions.check(EDIT_FOLDER, folderId);
        folderService.delete(folderId);
    }

    @Timed
    @Override
    public void move(String folderId, String newParentId) {
        checkArgument(!ROOT_FOLDER_ID.equals(folderId), "Cannot move root folder");
        permissions.check(EDIT_FOLDER, folderId);
        permissions.check(EDIT_FOLDER, newParentId);
        folderService.move(folderId, newParentId);
    }

    @Timed
    @Override
    public void rename(String folderId, String newName) {
        checkArgument(!ROOT_FOLDER_ID.equals(folderId), "Cannot rename root folder");
        permissions.check(EDIT_FOLDER, folderId);
        folderService.rename(folderId, newName);
    }

    @Timed
    @Override
    public List<String> getPermissions() {
        List<String> folderPermissions = new ArrayList<>();
        folderPermissions.addAll(XLReleasePermissions.getFolderPermissions());
        folderPermissions.addAll(XLReleasePermissions.getTemplatePermissions());
        return folderPermissions;
    }

    @Timed
    @Override
    public List<TeamView> getTeams(String folderId) {
        return teamFacade.getTeams(folderId);
    }

    @Timed
    @Override
    public List<TeamView> setTeams(String folderId, List<TeamView> teamDtos) {
        return teamFacade.setTeams(folderId, teamDtos);
    }

    @Timed
    @Override
    public boolean isFolderOwner(String folderId) {
        permissions.checkAny(folderId, EDIT_FOLDER_SECURITY, EDIT_FOLDER_TEAMS);
        return permissions.isCurrentUserAdmin() || permissions.getUserFolderTeams(folderId).stream().anyMatch(Team::isFolderOwnerTeam);
    }

    @Timed
    @Override
    public List<Variable> listVariables(String folderId, boolean folderOnly) {
        if (folderOnly) {
            return folderVariableService.getAllFromImmediateParent(folderId).getVariables();
        } else {
            return folderVariableService.getAllFromAncestry(folderId).getVariables();
        }
    }

    @Timed
    @Override
    public Map<String, String> listVariableValues(String folderId, boolean folderOnly) {
        return VariableHelper.getVariableValuesAsStrings(listVariables(folderId, folderOnly));
    }

    @Timed
    @Override
    public Variable createVariable(String folderId, com.xebialabs.xlrelease.api.v1.forms.Variable variable) {
        permissions.check(EDIT_FOLDER_VARIABLES, folderId);

        externalVariableService.checkExistsAndCorrectType(variable.getExternalVariableValue());

        Variable releaseVariable = variable.toReleaseVariable();
        releaseVariable.setFolderId(folderId);
        return folderVariableService.createFolderVariable(releaseVariable);
    }

    @Timed
    @Override
    public Variable getVariable(String folderId, String variableId) {
        return folderVariableService.findById(variableId);
    }

    @Timed
    @Override
    public Variable updateVariable(String folderId, String variableId, Variable variable) {
        permissions.check(EDIT_FOLDER_VARIABLES, folderId);
        variable.setId(variableId);
        variable.setFolderId(folderId);
        return folderVariableService.updateFolderVariable(variable);
    }

    @Timed
    @Override
    public Variable updateVariable(String folderId, Variable variable) {
        return updateVariable(folderId, variable.getId(), variable);
    }

    @Timed
    @Override
    public void deleteVariable(String folderId, String variableId) {
        permissions.check(EDIT_FOLDER_VARIABLES, folderId);
        folderVariableService.deleteFolderVariable(variableId);
    }

}

